/**
 * Utilitário para aplicação de máscara monetária
 * Versão corrigida que resolve o problema de travamento no formato #.##0,00
 * 
 * @author Sistema OutletByV
 * @version 2.0
 * @date 2024
 */

/**
 * Aplica máscara monetária corrigida em um campo específico
 * @param {string|jQuery} seletor - Seletor CSS ou objeto jQuery do campo
 * @param {object} opcoes - Opções adicionais para a máscara
 */
function aplicarMascaraMonetaria(seletor, opcoes = {}) {
    const campo = $(seletor);
    
    if (campo.length === 0) {
        console.warn('Campo não encontrado:', seletor);
        return;
    }
    
    // Configuração padrão da máscara corrigida
    const configPadrao = {
        reverse: true,
        placeholder: '0,00',
        translation: {
            '0': {pattern: /[0-9]/}
        }
    };
    
    // Mesclar configurações
    const config = Object.assign({}, configPadrao, opcoes);
    
    // Salvar valor atual
    const valorAtual = campo.val();
    
    // Remover máscara existente
    campo.unmask();
    
    // SOLUÇÃO DEFINITIVA: Implementação customizada sem jquery.mask
    // Remove qualquer máscara existente
    campo.unmask();
    
    // Função para formatar valor em tempo real
    function formatarValor(valor) {
        // Remove tudo que não é número
        valor = valor.replace(/\D/g, '');
        
        // Se vazio, retorna vazio
        if (!valor) return '';
        
        // Converte para centavos
        valor = parseInt(valor);
        
        // Formata como moeda brasileira
        return (valor / 100).toLocaleString('pt-BR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }
    
    // Evento de input para formatação em tempo real
    campo.off('input.mascara').on('input.mascara', function(e) {
        const cursorPos = this.selectionStart;
        const valorAnterior = $(this).val();
        const valorFormatado = formatarValor($(this).val());
        
        $(this).val(valorFormatado);
        
        // Ajustar posição do cursor
        const novaPosicao = Math.min(cursorPos + (valorFormatado.length - valorAnterior.length), valorFormatado.length);
        this.setSelectionRange(novaPosicao, novaPosicao);
    });
    
    // Evento de foco para seleção
    campo.off('focus.mascara').on('focus.mascara', function() {
        // Selecionar todo o conteúdo após um pequeno delay
        setTimeout(() => {
            this.select();
        }, 10);
    });
    
    // Permitir apenas números, vírgula, ponto e teclas de controle
    campo.off('keydown.mascara').on('keydown.mascara', function(e) {
        const permitidas = [
            8, 9, 27, 13, 46, // Backspace, Tab, Esc, Enter, Delete
            35, 36, 37, 38, 39, 40, // Home, End, setas
            96, 97, 98, 99, 100, 101, 102, 103, 104, 105, // Numpad 0-9
            48, 49, 50, 51, 52, 53, 54, 55, 56, 57 // 0-9
        ];
        
        // Permitir Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
        if ((e.ctrlKey || e.metaKey) && [65, 67, 86, 88].includes(e.keyCode)) {
            return true;
        }
        
        // Bloquear se não estiver na lista de permitidas
        if (!permitidas.includes(e.keyCode)) {
            e.preventDefault();
            return false;
        }
    });
    
    // Restaurar valor se existir
    if (valorAtual && valorAtual.trim() !== '') {
        // Limpar caracteres não numéricos para reprocessamento
        const valorLimpo = valorAtual.replace(/[^0-9,]/g, '');
        campo.val(valorLimpo);
        // Trigger para processar a máscara
        campo.trigger('input');
    }
    
    console.log('Máscara monetária aplicada em:', seletor);
}

/**
 * Aplica máscara monetária em múltiplos campos
 * @param {array} seletores - Array de seletores CSS
 * @param {object} opcoes - Opções adicionais para a máscara
 */
function aplicarMascaraMonetariaMultiplos(seletores, opcoes = {}) {
    seletores.forEach(function(seletor) {
        aplicarMascaraMonetaria(seletor, opcoes);
    });
}

/**
 * Remove máscara monetária de um campo
 * @param {string|jQuery} seletor - Seletor CSS ou objeto jQuery do campo
 */
function removerMascaraMonetaria(seletor) {
    const campo = $(seletor);
    
    if (campo.length > 0) {
        campo.unmask();
        console.log('Máscara monetária removida de:', seletor);
    }
}

/**
 * Converte valor com máscara para número
 * @param {string} valorMascarado - Valor com máscara (ex: "1.234,56")
 * @returns {number} - Valor numérico (ex: 1234.56)
 */
function converterMascaraParaNumero(valorMascarado) {
    if (!valorMascarado || typeof valorMascarado !== 'string') {
        return 0;
    }
    
    // Remove pontos e substitui vírgula por ponto
    const valorLimpo = valorMascarado
        .replace(/\./g, '')
        .replace(',', '.');
    
    return parseFloat(valorLimpo) || 0;
}

/**
 * Converte número para formato com máscara
 * @param {number} numero - Número a ser formatado
 * @returns {string} - Valor formatado (ex: "1.234,56")
 */
function converterNumeroParaMascara(numero) {
    if (typeof numero !== 'number' || isNaN(numero)) {
        return '0,00';
    }
    
    return numero.toLocaleString('pt-BR', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

/**
 * Valida se um valor monetário está no formato correto
 * @param {string} valor - Valor a ser validado
 * @returns {boolean} - True se válido, false caso contrário
 */
function validarValorMonetario(valor) {
    if (!valor || typeof valor !== 'string') {
        return false;
    }
    
    // Regex para formato brasileiro: 1.234,56 ou 1234,56 ou 1234.56
    const regex = /^\d{1,3}(\.\d{3})*,\d{2}$|^\d+,\d{2}$|^\d+\.\d{2}$/;
    return regex.test(valor.trim());
}

/**
 * Inicialização automática para campos com classe específica
 */
$(document).ready(function() {
    // Aplicar automaticamente em campos com classe 'mascara-monetaria'
    if ($('.mascara-monetaria').length > 0) {
        aplicarMascaraMonetaria('.mascara-monetaria');
        console.log('Máscaras monetárias aplicadas automaticamente');
    }
    
    // Aplicar automaticamente em campos com ID 'valor'
    if ($('#valor').length > 0) {
        aplicarMascaraMonetaria('#valor');
    }
});

// Exportar funções para uso global
window.aplicarMascaraMonetaria = aplicarMascaraMonetaria;
window.aplicarMascaraMonetariaMultiplos = aplicarMascaraMonetariaMultiplos;
window.removerMascaraMonetaria = removerMascaraMonetaria;
window.converterMascaraParaNumero = converterMascaraParaNumero;
window.converterNumeroParaMascara = converterNumeroParaMascara;
window.validarValorMonetario = validarValorMonetario;

/**
 * Log de inicialização
 */
console.log('Utilitário de Máscara Monetária v2.0 carregado');
console.log('Funções disponíveis:', {
    aplicarMascaraMonetaria: 'Aplica máscara em um campo',
    aplicarMascaraMonetariaMultiplos: 'Aplica máscara em múltiplos campos',
    removerMascaraMonetaria: 'Remove máscara de um campo',
    converterMascaraParaNumero: 'Converte valor mascarado para número',
    converterNumeroParaMascara: 'Converte número para formato mascarado',
    validarValorMonetario: 'Valida formato de valor monetário'
});