<?php
// Incluir configuração do banco de dados
require_once 'dbconfig.php';

// Configurar headers
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, must-revalidate');

// Log da requisição para debug
error_log('=== CRIAR EMPRESA DEBUG ===');
error_log('Method: ' . $_SERVER['REQUEST_METHOD']);
error_log('POST data: ' . print_r($_POST, true));
error_log('Headers: ' . print_r(getallheaders(), true));

// Verificar se é uma requisição POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Método não permitido. Use POST.',
        'debug_info' => [
            'method_received' => $_SERVER['REQUEST_METHOD'],
            'expected' => 'POST'
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Verificar se os parâmetros necessários foram enviados
if (!isset($_POST['nome_empresa']) || !isset($_POST['id_cliente'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Parâmetros obrigatórios não informados',
        'debug_info' => [
            'received_params' => array_keys($_POST),
            'required_params' => ['nome_empresa', 'id_cliente']
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$nome_empresa = trim($_POST['nome_empresa']);
$id_cliente = (int) $_POST['id_cliente'];

// Log dos parâmetros recebidos
error_log('Nome empresa: ' . $nome_empresa);
error_log('ID cliente: ' . $id_cliente);

// Validar dados
if (empty($nome_empresa)) {
    echo json_encode([
        'success' => false,
        'message' => 'Nome da empresa não pode estar vazio',
        'debug_info' => [
            'nome_empresa_raw' => $_POST['nome_empresa'],
            'nome_empresa_trimmed' => $nome_empresa
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if ($id_cliente <= 0) {
    echo json_encode([
        'success' => false,
        'message' => 'ID do cliente inválido',
        'debug_info' => [
            'id_cliente_raw' => $_POST['id_cliente'],
            'id_cliente_converted' => $id_cliente
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Validar tamanho do nome
if (strlen($nome_empresa) > 255) {
    echo json_encode([
        'success' => false,
        'message' => 'Nome da empresa muito longo (máximo 255 caracteres)',
        'debug_info' => [
            'nome_length' => strlen($nome_empresa),
            'max_length' => 255
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Verificar conexão com banco
if (!$conn) {
    error_log('Erro de conexão com banco: ' . mysqli_connect_error());
    echo json_encode([
        'success' => false,
        'message' => 'Erro de conexão com o banco de dados',
        'debug_info' => [
            'mysqli_error' => mysqli_connect_error()
        ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

try {
    // Verificar se o cliente existe
    $stmt = $conn->prepare("SELECT nome FROM usuarios WHERE id = ?");
    if (!$stmt) {
        throw new Exception('Erro ao preparar query de verificação do cliente: ' . $conn->error);
    }
    
    $stmt->bind_param("i", $id_cliente);
    if (!$stmt->execute()) {
        throw new Exception('Erro ao executar query de verificação do cliente: ' . $stmt->error);
    }
    
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode([
            'success' => false,
            'message' => 'Cliente não encontrado',
            'debug_info' => [
                'id_cliente_buscado' => $id_cliente
            ]
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    $cliente = $result->fetch_assoc();
    error_log('Cliente encontrado: ' . $cliente['nome']);
    
    // Verificar se já existe uma empresa com este nome
    $stmt = $conn->prepare("SELECT id, empresa FROM empresas WHERE empresa = ?");
    if (!$stmt) {
        throw new Exception('Erro ao preparar query de verificação da empresa: ' . $conn->error);
    }
    
    $stmt->bind_param("s", $nome_empresa);
    if (!$stmt->execute()) {
        throw new Exception('Erro ao executar query de verificação da empresa: ' . $stmt->error);
    }
    
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        // Empresa já existe, apenas associar ao cliente
        $empresa_existente = $result->fetch_assoc();
        error_log('Empresa já existe: ' . $empresa_existente['empresa'] . ' (ID: ' . $empresa_existente['id'] . ')');
        // Verificar se a coluna id_empresa existe na tabela usuarios
        $stmt_check = $conn->prepare("SHOW COLUMNS FROM usuarios LIKE 'id_empresa'");
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            // Coluna id_empresa existe, fazer update normal
            $stmt = $conn->prepare("UPDATE usuarios SET id_empresa = ? WHERE id = ?");
            if (!$stmt) {
                throw new Exception('Erro ao preparar query de associação: ' . $conn->error);
            }
            
            $stmt->bind_param("ii", $empresa_existente['id'], $id_cliente);
            
            if ($stmt->execute()) {
                error_log('Empresa existente associada com sucesso');
                echo json_encode([
                    'success' => true,
                    'message' => 'Empresa já existia e foi associada ao cliente',
                    'id_empresa' => $empresa_existente['id'],
                    'nome_empresa' => $empresa_existente['empresa'],
                    'cliente' => $cliente['nome'],
                    'empresa_ja_existia' => true
                ], JSON_UNESCAPED_UNICODE);
            } else {
                throw new Exception('Erro ao associar empresa existente ao cliente: ' . $stmt->error);
            }
        } else {
            // Coluna id_empresa não existe, apenas retornar sucesso da empresa encontrada
            error_log('Empresa existente encontrada (coluna id_empresa não disponível)');
            echo json_encode([
                'success' => true,
                'message' => 'Empresa já existia no sistema',
                'id_empresa' => $empresa_existente['id'],
                'nome_empresa' => $empresa_existente['empresa'],
                'cliente' => $cliente['nome'],
                'empresa_ja_existia' => true,
                'observacao' => 'Empresa não foi associada automaticamente (coluna id_empresa não existe)'
            ], JSON_UNESCAPED_UNICODE);
        }
    } else {
        // Iniciar transação para criar empresa e associar ao cliente
        if (!$conn->autocommit(false)) {
            throw new Exception('Erro ao iniciar transação: ' . $conn->error);
        }
        
        try {
            // Criar nova empresa
            $data_atual = date("d/m/Y");
            $hora_atual = date("H:i");
            
            error_log("Tentando criar empresa: nome='$nome_empresa', data='$data_atual', hora='$hora_atual'");
            
            $stmt = $conn->prepare("INSERT INTO empresas (empresa, data, hora) VALUES (?, ?, ?)");
            if (!$stmt) {
                error_log("ERRO ao preparar query de inserção: " . $conn->error);
                throw new Exception('Erro ao preparar query de criação da empresa: ' . $conn->error);
            }
            
            error_log("Query preparada com sucesso, fazendo bind dos parâmetros...");
            $stmt->bind_param("sss", $nome_empresa, $data_atual, $hora_atual);
            
            error_log("Parâmetros vinculados, executando query...");
            if (!$stmt->execute()) {
                error_log("ERRO ao executar query de inserção: " . $stmt->error);
                throw new Exception('Erro ao criar empresa: ' . $stmt->error);
            }
            
            $nova_empresa_id = $conn->insert_id;
            error_log("Nova empresa criada com ID: $nova_empresa_id");
            
            // Associar empresa ao cliente (verificar se a coluna id_empresa existe)
            $stmt_check = $conn->prepare("SHOW COLUMNS FROM usuarios LIKE 'id_empresa'");
            $stmt_check->execute();
            $result_check = $stmt_check->get_result();
            
            if ($result_check->num_rows > 0) {
                // Coluna id_empresa existe, fazer update normal
                $stmt = $conn->prepare("UPDATE usuarios SET id_empresa = ? WHERE id = ?");
                if (!$stmt) {
                    throw new Exception('Erro ao preparar query de associação: ' . $conn->error);
                }
                
                $stmt->bind_param("ii", $nova_empresa_id, $id_cliente);
                
                if (!$stmt->execute()) {
                    throw new Exception('Erro ao associar empresa ao cliente: ' . $stmt->error);
                }
                
                error_log("Empresa associada ao cliente via coluna id_empresa");
            } else {
                // Coluna id_empresa não existe, apenas log do sucesso
                error_log("Nota: Coluna id_empresa não existe na tabela usuarios. Empresa criada mas não associada automaticamente.");
            }
            
            // Confirmar transação
            if (!$conn->commit()) {
                throw new Exception('Erro ao confirmar transação: ' . $conn->error);
            }
            
            $conn->autocommit(true);
            
            // Log da operação
            error_log("Nova empresa criada: '{$nome_empresa}' (ID: {$nova_empresa_id}) e associada ao cliente {$cliente['nome']} (ID: {$id_cliente})");
            
            echo json_encode([
                'success' => true,
                'message' => 'Empresa criada e associada com sucesso',
                'id_empresa' => $nova_empresa_id,
                'nome_empresa' => $nome_empresa,
                'cliente' => $cliente['nome'],
                'empresa_ja_existia' => false
            ], JSON_UNESCAPED_UNICODE);
            
        } catch (Exception $e) {
            // Reverter transação
            $conn->rollback();
            $conn->autocommit(true);
            throw $e;
        }
    }
    
} catch (Exception $e) {
    error_log('Erro ao criar/associar empresa: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    
    // Verificar se é erro de conexão com banco
    if (strpos($e->getMessage(), 'Connection') !== false || strpos($e->getMessage(), 'conexão') !== false) {
        echo json_encode([
            'success' => false,
            'message' => 'Erro de conexão com o banco de dados. Verifique se o servidor está ativo.',
            'debug_info' => [
                'exception_message' => $e->getMessage(),
                'exception_file' => $e->getFile(),
                'exception_line' => $e->getLine(),
                'tipo_erro' => 'conexao_banco'
            ]
        ], JSON_UNESCAPED_UNICODE);
    } 
    // Verificar se é erro de query/SQL
    else if (strpos($e->getMessage(), 'preparar query') !== false || strpos($e->getMessage(), 'SQL') !== false) {
        echo json_encode([
            'success' => false,
            'message' => 'Erro na estrutura do banco de dados. Verifique se as tabelas estão corretas.',
            'debug_info' => [
                'exception_message' => $e->getMessage(),
                'exception_file' => $e->getFile(),
                'exception_line' => $e->getLine(),
                'tipo_erro' => 'estrutura_banco'
            ]
        ], JSON_UNESCAPED_UNICODE);
    }
    // Erro genérico
    else {
        echo json_encode([
            'success' => false,
            'message' => 'Erro interno do servidor: ' . $e->getMessage(),
            'debug_info' => [
                'exception_message' => $e->getMessage(),
                'exception_file' => $e->getFile(),
                'exception_line' => $e->getLine(),
                'tipo_erro' => 'generico'
            ]
        ], JSON_UNESCAPED_UNICODE);
    }
}
?>
