<?php
// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once 'dbconfig.php';
include_once 'includes/antisql.php';

// Verificar se a conexão foi estabelecida
if (!$conn) {
    echo json_encode(['success' => false, 'message' => 'Erro de conexão com banco de dados: ' . mysqli_connect_error()]);
    exit;
}

// Processar ações via POST - DEVE estar no início antes de qualquer saída HTML
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // LOG DE DEBUG - INÍCIO
    error_log("=== FINANCEIRO_CLIENTES_MODAL.PHP - REQUISIÇÃO POST RECEBIDA ===");
    error_log("POST DATA: " . print_r($_POST, true));
    error_log("TIMESTAMP: " . date('Y-m-d H:i:s'));
    error_log("USER_AGENT: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'N/A'));
    error_log("REFERER: " . ($_SERVER['HTTP_REFERER'] ?? 'N/A'));
    // LOG DE DEBUG - FIM
    
    header('Content-Type: application/json');
    
    $acao = isset($_POST['acao']) ? $_POST['acao'] : '';
    $id_pedido = isset($_POST['id_pedido']) ? (int)$_POST['id_pedido'] : 0;
    $id_cliente = isset($_POST['id_cliente']) ? (int)$_POST['id_cliente'] : 0;
    $id_financeiro_clientes = isset($_POST['id_financeiro_clientes']) ? (int)$_POST['id_financeiro_clientes'] : 0;
    
    if ($acao === 'inserir') {
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : '0';
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        // Log de debug
        error_log("DEBUG INSERIR: valor=$valor, data=$data, id_pedido=$id_pedido, id_cliente=$id_cliente");
        
        // Validações melhoradas
        if (empty($valor) || !is_numeric($valor) || floatval($valor) <= 0) {
            echo json_encode(['success' => false, 'message' => 'Valor deve ser um número maior que zero.', 'debug' => "valor='$valor', numeric=" . (is_numeric($valor) ? 'true' : 'false')]);
            exit;
        }
        
        if (empty($data) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
            echo json_encode(['success' => false, 'message' => 'Data deve estar no formato válido (YYYY-MM-DD).']);
            exit;
        }
        
        if ($id_pedido <= 0) {
            echo json_encode(['success' => false, 'message' => 'ID do pedido é obrigatório.']);
            exit;
        }
        
        // Converter valor para float
        $valor_float = floatval($valor);
        
        // Preparar datas
        $data_atual = date("d/m/Y"); // Data de criação do registro
        $hora_atual = date("H:i");   // Hora de criação do registro
        
        // Converter data do formulário para formato brasileiro se necessário
        $data_pagamento = $data; // Data vem no formato YYYY-MM-DD do input date
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
            $timestamp = strtotime($data);
            $data_pagamento = date('d/m/Y', $timestamp); // Converter para formato brasileiro
        }
        
        $sql = "INSERT INTO financeiro_clientes (id_pedido, valor, data, data2, hora) VALUES (?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("idsss", $id_pedido, $valor_float, $data_atual, $data_pagamento, $hora_atual);
            
            if ($stmt->execute()) {
                $insert_id = $conn->insert_id;
                error_log("DEBUG INSERIR SUCCESS: insert_id=$insert_id, data_atual=$data_atual, data_pagamento=$data_pagamento, hora=$hora_atual");
                echo json_encode(['success' => true, 'message' => 'Pagamento adicionado com sucesso!', 'id_financeiro_clientes' => $insert_id]);
            } else {
                error_log("DEBUG INSERIR ERROR: " . $stmt->error);
                echo json_encode(['success' => false, 'message' => 'Erro ao adicionar pagamento: ' . $stmt->error]);
            }
        } else {
            error_log("DEBUG PREPARE ERROR: " . $conn->error);
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    } elseif ($acao === 'alterar' && $id_financeiro_clientes > 0) {
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : '0';
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        // Verificar se o registro existe antes de tentar atualizar
        $check_sql = "SELECT id FROM financeiro_clientes WHERE id = ?";
        $check_stmt = $conn->prepare($check_sql);
        if ($check_stmt) {
            $check_stmt->bind_param("i", $id_financeiro_clientes);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            if ($check_result->num_rows === 0) {
                echo json_encode(['success' => false, 'message' => "Registro com ID $id_financeiro_clientes não encontrado no banco de dados."]);
                exit;
            }
        }
        
        if ($valor > 0) {
            // Converter data do formulário para formato brasileiro se necessário
            $data_pagamento = $data;
            if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
                $timestamp = strtotime($data);
                $data_pagamento = date('d/m/Y', $timestamp);
            }
            
            $sql = "UPDATE financeiro_clientes SET valor = ?, data2 = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("ssi", $valor, $data_pagamento, $id_financeiro_clientes);
                
                if ($stmt->execute()) {
                    // Verificar se alguma linha foi afetada
                    if ($stmt->affected_rows > 0) {
                        echo json_encode(['success' => true, 'message' => 'Pagamento atualizado com sucesso!', 'debug' => "Valor: $valor, Data: $data"]);
                    } else {
                        echo json_encode(['success' => false, 'message' => 'Nenhuma linha foi atualizada. Verifique se o ID existe.']);
                    }
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao atualizar pagamento: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Valor é obrigatório.']);
            exit;
        }
    } elseif ($acao === 'excluir' && $id_financeiro_clientes > 0) {
        $sql = "DELETE FROM financeiro_clientes WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_financeiro_clientes);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Pagamento excluído com sucesso!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao excluir pagamento: ' . $stmt->error]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    } elseif ($acao === 'alterar_cliente' && $id_cliente > 0) {
        $nome_cliente = isset($_POST['nome_cliente']) ? trim($_POST['nome_cliente']) : '';
        
        // Validações
        if (empty($nome_cliente)) {
            echo json_encode(['success' => false, 'message' => 'Nome do cliente é obrigatório.']);
            exit;
        }
        
        if (strlen($nome_cliente) < 2) {
            echo json_encode(['success' => false, 'message' => 'Nome do cliente deve ter pelo menos 2 caracteres.']);
            exit;
        }
        
        if (strlen($nome_cliente) > 100) {
            echo json_encode(['success' => false, 'message' => 'Nome do cliente deve ter no máximo 100 caracteres.']);
            exit;
        }
        
        // Verificar se o cliente existe
        $check_sql = "SELECT id, nome FROM usuarios WHERE id = ?";
        $check_stmt = $conn->prepare($check_sql);
        if ($check_stmt) {
            $check_stmt->bind_param("i", $id_cliente);
            $check_stmt->execute();
            $check_result = $check_stmt->get_result();
            if ($check_result->num_rows === 0) {
                echo json_encode(['success' => false, 'message' => "Cliente com ID $id_cliente não encontrado."]);
                exit;
            }
            $cliente_atual = $check_result->fetch_assoc();
        }
        
        // Atualizar nome do cliente
        $sql = "UPDATE usuarios SET nome = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("si", $nome_cliente, $id_cliente);
            
            if ($stmt->execute()) {
                if ($stmt->affected_rows > 0) {
                    error_log("DEBUG ALTERAR CLIENTE: Nome alterado de '{$cliente_atual['nome']}' para '$nome_cliente' (ID: $id_cliente)");
                    echo json_encode([
                        'success' => true, 
                        'message' => 'Nome do cliente atualizado com sucesso!',
                        'nome_cliente' => $nome_cliente
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Nenhuma alteração foi feita. O nome pode já ser o mesmo.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao atualizar nome do cliente: ' . $stmt->error]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    }
}

// Processar ação buscar_pagamento via GET
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['acao']) && $_GET['acao'] === 'buscar_pagamento') {
    header('Content-Type: application/json');
    
    $id_financeiro_clientes = isset($_GET['id_financeiro_clientes']) ? (int)$_GET['id_financeiro_clientes'] : 0;
    
    if ($id_financeiro_clientes > 0) {
        $sql = "SELECT * FROM financeiro_clientes WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_financeiro_clientes);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $pagamento = $result->fetch_assoc();
                
                if ($pagamento) {
                    $valor_formatado = number_format($pagamento['valor'], 2, ',', '.');
                    
                    echo json_encode([
                        'success' => true,
                        'data' => [
                            'valor_formatado' => $valor_formatado,
                            'data_pagamento' => $pagamento['data2']
                        ]
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Pagamento não encontrado.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao buscar pagamento.']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID do pagamento inválido.']);
    }
    exit;
}

// Receber parâmetros
$id_pedido = isset($_GET['id_pedido']) ? (int)$_GET['id_pedido'] : 0;
$id_cliente = isset($_GET['id_cliente']) ? (int)$_GET['id_cliente'] : 0;

// Debug dos parâmetros recebidos
error_log("DEBUG MODAL FINANCEIRO - Parâmetros: id_pedido=$id_pedido, id_cliente=$id_cliente");

// Buscar informações do pedido e cliente
$info_pedido = null;
if ($id_pedido > 0 && $id_cliente > 0) {
    $sql = "SELECT p.id as pedido_id, p.data, u.nome as cliente_nome 
            FROM pedidos p 
            LEFT JOIN usuarios u ON p.id_cliente = u.id 
            WHERE p.id = ? AND p.id_cliente = ?";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("ii", $id_pedido, $id_cliente);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $info_pedido = $result->fetch_assoc();
            error_log("DEBUG MODAL FINANCEIRO - Info pedido encontrada: " . ($info_pedido ? 'SIM' : 'NÃO'));
        }
    }
}

// Buscar pagamentos do pedido
$pagamentos = [];
if ($id_pedido > 0) {
    $sql = "SELECT * FROM financeiro_clientes WHERE id_pedido = ? ORDER BY id DESC";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("i", $id_pedido);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $pagamentos[] = $row;
            }
            error_log("DEBUG MODAL FINANCEIRO - Pagamentos encontrados: " . count($pagamentos));
        } else {
            error_log("DEBUG MODAL FINANCEIRO - Erro ao executar query de pagamentos: " . $stmt->error);
        }
    } else {
        error_log("DEBUG MODAL FINANCEIRO - Erro ao preparar query de pagamentos: " . $conn->error);
    }
} else {
    error_log("DEBUG MODAL FINANCEIRO - ID pedido inválido para buscar pagamentos: $id_pedido");
}

// Calcular total de pagamentos
$total_pagamentos = 0;
foreach ($pagamentos as $pagamento) {
    $total_pagamentos += $pagamento['valor'];
}

// Função para formatar data
function formatarData($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return 'Sem data';
    }
    
    // Se já está no formato dd/mm/yyyy
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        return $data;
    }
    
    // Se está no formato yyyy-mm-dd
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        $timestamp = strtotime($data);
        if ($timestamp && $timestamp > 0) {
            return date('d/m/Y', $timestamp);
        }
    }
    
    return $data;
}

// Função para converter data para input date
function converterDataParaInput($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return '';
    }
    
    // Se já está no formato yyyy-mm-dd
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        return $data;
    }
    
    // Se está no formato dd/mm/yyyy
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        $partes = explode('/', $data);
        return $partes[2] . '-' . $partes[1] . '-' . $partes[0];
    }
    
    // Tentar converter outros formatos
    $timestamp = strtotime($data);
    if ($timestamp && $timestamp > 0) {
        return date('Y-m-d', $timestamp);
    }
    
    return '';
}
?>

<!-- SweetAlert2 -->
<link href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css" rel="stylesheet">
<!-- SweetAlert2 JS -->
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<div class="container-fluid mt-3">
    <!-- Informações do Pedido -->
    <?php if ($info_pedido): ?>
    <div class="row mb-3" id="info-pedido-container">
        <div class="col-12">
            <div class="alert alert-info" id="info-pedido-alert">
                <strong>Pedido ID:</strong> <?php echo htmlspecialchars($info_pedido['pedido_id']); ?> | 
                <strong>Data:</strong> <?php echo formatarData($info_pedido['data']); ?> |
                <strong>Cliente:</strong> 
                <span class="cliente-nome-display" data-original="<?php echo htmlspecialchars($info_pedido['cliente_nome']); ?>" onclick="FinanceiroClientesModal.editarNomeCliente()">
                    <?php echo htmlspecialchars($info_pedido['cliente_nome']); ?>
                    <i class="fas fa-edit ml-1" style="font-size: 12px; color: #007bff; cursor: pointer;" title="Clique para editar"></i>
                </span>
                <input type="text" class="cliente-nome-input form-control d-inline-block" style="display: none; width: 200px; margin-left: 5px;" value="<?php echo htmlspecialchars($info_pedido['cliente_nome']); ?>">
                <button class="btn btn-sm btn-success cliente-salvar-btn" style="display: none; margin-left: 5px;" onclick="FinanceiroClientesModal.salvarNomeCliente()">
                    <i class="fas fa-check"></i>
                </button>
                <button class="btn btn-sm btn-secondary cliente-cancelar-btn" style="display: none; margin-left: 5px;" onclick="FinanceiroClientesModal.cancelarEdicaoNomeCliente()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Mensagens de alerta -->
    <div class="row mb-4">
        <div class="col-12">
            <div id="alertMessages"></div>
        </div>
    </div>

    <!-- Lista de Pagamentos -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Pagamentos do Pedido</h5>
                    <div class="d-flex align-items-center gap-2">
                        <button type="button" class="btn btn-success btn-sm" onclick="FinanceiroClientesModal.adicionarNovaLinha()" title="Adicionar Novo Pagamento">
                            <i class="fas fa-plus"></i> Novo Pagamento
                        </button>
                        <span class="badge bg-primary fs-6">Total: R$ <?php echo number_format($total_pagamentos, 2, ',', '.'); ?></span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-hover" id="tabelaPagamentos">
                            <thead class="table-dark">
                                <tr>
                                    <th>Nº</th>
                                    <th>Valor</th>
                                    <th>Data</th>
                                    <th width="120">Ações</th>
                                </tr>
                            </thead>
                            <tfoot>
                                <tr>
                                    <th>Nº</th>
                                    <th>Valor</th>
                                    <th>Data</th>
                                    <th>Ações</th>
                                </tr>
                            </tfoot>
                            <tbody>
                                <?php if (empty($pagamentos)): ?>
                                <!-- Tabela vazia, pronta para receber novos registros -->
                                <?php else: ?>
                                <?php foreach ($pagamentos as $pagamento): ?>
                                <tr data-id="<?php echo $pagamento['id']; ?>">
                                    <td><?php echo $pagamento['id']; ?></td>
                                    <td class="editable-cell" data-field="valor" data-original="<?php echo $pagamento['valor']; ?>">
                                        <span class="cell-display">R$ <?php echo number_format($pagamento['valor'], 2, ',', '.'); ?></span>
                                        <input type="text" class="cell-input form-control form-control-sm valor-input" value="<?php echo number_format($pagamento['valor'], 2, ',', '.'); ?>" style="display: none;">
                                    </td>
                                    <td class="editable-cell" data-field="data2" data-original="<?php echo $pagamento['data2']; ?>">
                                        <span class="cell-display"><?php echo formatarData($pagamento['data2']); ?></span>
                                        <input type="date" class="cell-input form-control form-control-sm" value="<?php echo converterDataParaInput($pagamento['data2']); ?>" style="display: none;">
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-success btn-save" onclick="FinanceiroClientesModal.salvarEdicaoInline(<?php echo $pagamento['id']; ?>)" title="Salvar" style="display: none;">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="FinanceiroClientesModal.cancelarEdicaoInline(<?php echo $pagamento['id']; ?>)" title="Cancelar" style="display: none;">
                                            <i class="fas fa-times"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-warning btn-edit" onclick="FinanceiroClientesModal.editarInline(<?php echo $pagamento['id']; ?>)" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button type="button" class="btn btn-sm btn-danger" onclick="FinanceiroClientesModal.excluirPagamento(<?php echo $pagamento['id']; ?>)" title="Excluir">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <?php if (empty($pagamentos)): ?>
                    <div class="text-center text-muted py-4" id="mensagem-vazio">
                        <i class="fas fa-money-bill-wave fa-3x mb-3"></i>
                        <p>Nenhum pagamento encontrado para este pedido.<br>
                        <small>Clique em "Novo Pagamento" para adicionar o primeiro registro.</small></p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- CSS Específico do Modal -->

<style>
/* Proteção para informações do pedido - garantir que sempre fiquem visíveis */
#info-pedido-container {
    position: relative !important;
    z-index: 1000 !important;
    display: block !important;
    visibility: visible !important;
    opacity: 1 !important;
    margin-bottom: 1rem !important;
}

#info-pedido-alert {
    display: block !important;
    visibility: visible !important;
    opacity: 1 !important;
    margin-bottom: 0 !important;
}

/* Proteção adicional com seletores mais específicos */
.container-fluid #info-pedido-container {
    display: block !important;
}

.container-fluid #info-pedido-alert {
    display: block !important;
}

/* Prevenir que DataTables ou outros scripts removam as informações */
#info-pedido-container[style*="display: none"] {
    display: block !important;
}

#info-pedido-alert[style*="display: none"] {
    display: block !important;
}

/* Estilos mínimos específicos para edição inline */
.editable-cell { cursor: pointer; position: relative; }
.editable-cell:hover { background-color: #f8f9fa; }
.editable-cell.editing { background-color: #fff3cd; cursor: default; }
.cell-input { width: 100%; border: 2px solid #007bff; background-color: #fff; }
.cell-input:focus { border-color: #0056b3; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
.valor-input { text-align: right; }
.editing-row { background-color: #fff3cd !important; }
.nova-linha { background-color: #d1ecf1 !important; }

/* Estilos para edição inline do nome do cliente */
.cliente-nome-display {
    cursor: pointer;
    padding: 2px 6px;
    border-radius: 4px;
    transition: background-color 0.2s ease;
    display: inline-block;
}

.cliente-nome-display:hover {
    background-color: rgba(0, 123, 255, 0.1);
    color: #0056b3;
}

.cliente-nome-input {
    border: 2px solid #007bff !important;
    background-color: #fff !important;
    font-weight: normal !important;
    font-size: inherit !important;
}

.cliente-nome-input:focus {
    border-color: #0056b3 !important;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25) !important;
    outline: none !important;
}

.cliente-salvar-btn, .cliente-cancelar-btn {
    border-radius: 4px;
    padding: 4px 8px;
    font-size: 12px;
    line-height: 1;
    vertical-align: middle;
}

/* Estilos para os filtros de coluna */
.md-input-wrapper { margin-bottom: 0; }
.footer-search-input-financeiro {

}
.footer-search-input-financeiro:focus {

}
#tabelaPagamentos tfoot { background-color: #f8f9fa; }
#tabelaPagamentos tfoot th { padding: 8px 4px; font-weight: normal; }

/* Esconder busca global para evitar paginação duplicada */
#tabelaPagamentos_filter { display: none !important; }
.dataTables_filter { display: none !important; }

/* Forçar controles únicos do DataTable */
#tabelaPagamentos_wrapper .dataTables_paginate { display: block !important; }
#tabelaPagamentos_wrapper .dataTables_info { display: block !important; }
#tabelaPagamentos_wrapper .dataTables_length { display: block !important; }

#tabelaPagamentos { width: 100%; }
#tabelaPagamentos th, #tabelaPagamentos td { padding: 8px 12px; }
.table-hover tbody tr:hover { background-color: rgba(0, 0, 0, 0.075); }

/* Garantir que SweetAlert2 apareça na frente dos modais */
.swal2-container {
    z-index: 9999999 !important;
    position: fixed !important;
}

.swal2-popup {
    z-index: 9999999 !important;
    position: relative !important;
}

.swal2-backdrop {
    z-index: 9999998 !important;
}

/* Sobrescrever qualquer z-index de modais do Bootstrap */
.modal {
    z-index: 1050 !important;
}

.modal-backdrop {
    z-index: 1040 !important;
}

/* Garantir que o SweetAlert2 tenha a maior prioridade */
body .swal2-container {
    z-index: 9999999 !important;
}
</style>

<!-- JavaScript -->

<script>
// Namespace para evitar conflitos com outros modais
window.FinanceiroClientesModal = window.FinanceiroClientesModal || {};

// Variáveis globais do modal financeiro
FinanceiroClientesModal.ID_PEDIDO = <?php echo $id_pedido; ?>;
FinanceiroClientesModal.ID_CLIENTE = <?php echo $id_cliente; ?>;

// Função personalizada para SweetAlert2 com z-index forçado
FinanceiroClientesModal.showAlert = function(options) {
    // Configurações padrão com z-index forçado
    const defaultOptions = {
        customClass: {
            container: 'swal2-top-modal'
        },
        didOpen: function() {
            // Forçar z-index máximo
            const container = document.querySelector('.swal2-container');
            const popup = document.querySelector('.swal2-popup');
            const backdrop = document.querySelector('.swal2-backdrop');
            
            if (container) container.style.zIndex = '9999999';
            if (popup) popup.style.zIndex = '9999999';
            if (backdrop) backdrop.style.zIndex = '9999998';
        }
    };
    
    // Mesclar opções
    const finalOptions = Object.assign({}, defaultOptions, options);
    
    return Swal.fire(finalOptions);
};

// Função de debug
FinanceiroClientesModal.debug = function() {
    console.log('🔍 DEBUG MODAL FINANCEIRO');
    console.log('📊 ID_PEDIDO:', FinanceiroClientesModal.ID_PEDIDO);
    console.log('📊 ID_CLIENTE:', FinanceiroClientesModal.ID_CLIENTE);
    console.log('📊 Tabela existe:', $('#tabelaPagamentos').length > 0);
    console.log('📊 Linhas na tabela:', $('#tabelaPagamentos tbody tr').length);
    console.log('📊 DataTable ativo:', $.fn.DataTable.isDataTable('#tabelaPagamentos'));
    console.log('📊 Botão novo pagamento:', $('button[onclick*="adicionarNovaLinha"]').length);
    console.log('📊 Mensagem vazio visível:', $('#mensagem-vazio').is(':visible'));
    
    // Testar função de adicionar nova linha diretamente
    console.log('🧪 Testando adicionarNovaLinha...');
    try {
        FinanceiroClientesModal.adicionarNovaLinha();
        console.log('✅ Função adicionarNovaLinha executada');
    } catch (e) {
        console.log('❌ Erro ao executar adicionarNovaLinha:', e);
    }
};

// Função para forçar inicialização (pode ser chamada externamente)
FinanceiroClientesModal.forcarInicializacao = function() {
    console.log('🔄 Forçando inicialização do modal financeiro...');
    
    // Preservar informações do pedido antes da limpeza
    const infoHTML = $('#info-pedido-container').length > 0 ? $('#info-pedido-container')[0].outerHTML : null;
    
    // Preservar dados da tabela antes da limpeza
    FinanceiroClientesModal.preservarDadosTabela();
    
    // Limpar estado anterior
    window.financeiroModalCarregado = false;
    
    // Destruir DataTable se existir
    if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
        try {
            $('#tabelaPagamentos').DataTable().destroy(false);
        } catch (e) {
            console.log('⚠️ Erro ao destruir DataTable:', e);
        }
    }
    
    // Limpar DOM mas preservar informações do pedido
    $('.dataTables_wrapper').remove();
    $('#tabelaPagamentos').removeClass('dataTable');
    $('#tabelaPagamentos').removeAttr('role aria-describedby');
    
    // Restaurar informações do pedido se foram perdidas
    if (infoHTML && $('#info-pedido-container').length === 0) {
        console.log('🔄 Restaurando informações do pedido...');
        $('.container-fluid').prepend(infoHTML);
    }
    
    // Restaurar dados da tabela se foram perdidos
    FinanceiroClientesModal.restaurarDadosTabela();
    
    // Proteger informações imediatamente
    FinanceiroClientesModal.protegerInfoPedido();
    
    // Inicializar novamente
    setTimeout(function() {
        inicializarModalFinanceiro();
    }, 100);
};

// Função para reinicializar o modal (chamada quando abre o modal)
FinanceiroClientesModal.reinicializar = function() {
    console.log('🔄 Reinicializando modal financeiro...');
    
    // Resetar flag de carregamento
    window.financeiroModalCarregado = false;
    
    // Destruir DataTable se existir
    if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
        try {
            $('#tabelaPagamentos').DataTable().destroy(false);
            console.log('✅ DataTable destruído preservando conteúdo');
        } catch (e) {
            console.log('⚠️ Erro ao destruir DataTable:', e);
        }
    }
    
    // Limpar wrappers e elementos residuais
    $('.dataTables_wrapper').remove();
    $('#tabelaPagamentos').removeClass('dataTable');
    $('#tabelaPagamentos').removeAttr('role aria-describedby');
    
    // Limpar timeouts pendentes
    for (let i = 0; i < 10; i++) {
        clearTimeout(window['timeout_financeiro_col_' + i]);
    }
    
    // Forçar reinicialização
    setTimeout(function() {
        console.log('🚀 Forçando reinicialização...');
        $(document).trigger('ready');
    }, 100);
};

// Função para proteger informações do pedido
FinanceiroClientesModal.protegerInfoPedido = function() {
    console.log('🛡️ Protegendo informações do pedido...');
    
    // Função interna para forçar visibilidade
    function forcarVisibilidade() {
        const $infoPedido = $('#info-pedido-container');
        const $infoAlert = $('#info-pedido-alert');
        
        if ($infoPedido.length > 0) {
            // Forçar visibilidade com CSS inline
            $infoPedido[0].style.setProperty('display', 'block', 'important');
            $infoPedido[0].style.setProperty('visibility', 'visible', 'important');
            $infoPedido[0].style.setProperty('opacity', '1', 'important');
            $infoPedido[0].style.setProperty('position', 'relative', 'important');
            $infoPedido[0].style.setProperty('z-index', '1000', 'important');
            
            if ($infoAlert.length > 0) {
                $infoAlert[0].style.setProperty('display', 'block', 'important');
                $infoAlert[0].style.setProperty('visibility', 'visible', 'important');
                $infoAlert[0].style.setProperty('opacity', '1', 'important');
            }
            
            console.log('✅ Informações do pedido protegidas e visíveis');
            return true;
        } else {
            console.log('⚠️ Container de informações do pedido não encontrado');
            return false;
        }
    }
    
    // Primeira proteção imediata
    forcarVisibilidade();
    
    // Proteção com timeout curto
    setTimeout(forcarVisibilidade, 50);
    
    // Proteção com timeout médio
    setTimeout(forcarVisibilidade, 200);
    
    // Proteção com timeout longo
    setTimeout(forcarVisibilidade, 500);
    
    // Criar observer para monitorar mudanças no DOM
    if (typeof MutationObserver !== 'undefined') {
        // Remover observer anterior se existir
        if (FinanceiroClientesModal.infoProtectionObserver) {
            FinanceiroClientesModal.infoProtectionObserver.disconnect();
        }
        
        FinanceiroClientesModal.infoProtectionObserver = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                // Verificar se o container foi removido ou ocultado
                if (mutation.type === 'childList' || mutation.type === 'attributes') {
                    const $infoPedido = $('#info-pedido-container');
                    if ($infoPedido.length === 0 || !$infoPedido.is(':visible')) {
                        console.log('� Container de informações foi alterado, restaurando...');
                        setTimeout(forcarVisibilidade, 10);
                    }
                }
            });
        });
        
        // Observar mudanças no container modal
        const modalContainer = document.querySelector('.container-fluid');
        if (modalContainer) {
            FinanceiroClientesModal.infoProtectionObserver.observe(modalContainer, {
                childList: true,
                subtree: true,
                attributes: true,
                attributeFilter: ['style', 'class']
            });
        }
    }
};

// Função para aplicar máscara de valor
FinanceiroClientesModal.aplicarMascaraValorInline = function(campo) {
    campo.off('.mascaraInline');
    
    campo.on('input.mascaraInline', function() {
        let valor = this.value.replace(/\D/g, '');
        if (valor.length === 0) { 
            this.value = ''; 
            return; 
        }
        
        // Remover zeros à esquerda, mas manter pelo menos um dígito
        valor = valor.replace(/^0+/, '') || '0';
        
        // Aplicar lógica de formatação baseada no comprimento
        if (valor.length === 1) {
            // 1 dígito: centavos (2 -> 0,02)
            this.value = '0,0' + valor;
        } else if (valor.length === 2) {
            // 2 dígitos: centavos (20 -> 0,20)
            this.value = '0,' + valor;
        } else if (valor.length === 3) {
            // 3 dígitos: primeiro dígito como real, últimos dois como centavos (200 -> 2,00)
            this.value = valor.charAt(0) + ',' + valor.slice(1);
        } else {
            // 4+ dígitos: últimos dois como centavos
            const parteDecimal = valor.slice(-2);
            let parteInteira = valor.slice(0, -2);
            
            // Adicionar pontos para milhares se necessário
            if (parteInteira.length > 3) {
                parteInteira = parteInteira.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
            }
            
            this.value = parteInteira + ',' + parteDecimal;
        }
    });
    
    // Permitir backspace e delete funcionarem corretamente
    campo.on('keydown.mascaraInline', function(e) {
        // Permitir teclas especiais (backspace, delete, tab, escape, enter, etc.)
        if ([46, 8, 9, 27, 13].indexOf(e.keyCode) !== -1 ||
            // Permitir Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
            (e.keyCode === 65 && e.ctrlKey === true) ||
            (e.keyCode === 67 && e.ctrlKey === true) ||
            (e.keyCode === 86 && e.ctrlKey === true) ||
            (e.keyCode === 88 && e.ctrlKey === true)) {
            return;
        }
        
        // Permitir apenas números
        if ((e.shiftKey || (e.keyCode < 48 || e.keyCode > 57)) && (e.keyCode < 96 || e.keyCode > 105)) {
            e.preventDefault();
        }
    });
    
    campo.on('focus.mascaraInline', function() {
        // Selecionar todo o texto ao focar
        setTimeout(() => this.select(), 10);
    });
    
    // Tratar colagem de texto
    campo.on('paste.mascaraInline', function(e) {
        setTimeout(() => {
            let valor = this.value.replace(/\D/g, '');
            if (valor) {
                $(this).val(valor).trigger('input');
            }
        }, 10);
    });
}

// Função para cancelar todas as edições
FinanceiroClientesModal.cancelarTodasEdicoes = function() {
    $('.editing-row').each(function() {
        const idPagamento = $(this).data('id');
        if (idPagamento && !String(idPagamento).startsWith('novo_')) {
            FinanceiroClientesModal.cancelarEdicaoInline(idPagamento);
        }
    });
}

// Função para editar linha inline
FinanceiroClientesModal.editarInline = function(idPagamento) {
    FinanceiroClientesModal.cancelarTodasEdicoes();
    
    const linha = $(`tr[data-id="${idPagamento}"]`);
    linha.addClass('editing-row');
    
    // Preparar o valor antes de mostrar o input
    const valorCell = linha.find('[data-field="valor"]');
    const valorOriginal = valorCell.data('original');
    const valorInput = valorCell.find('.valor-input');
    
    // Formatar o valor para exibição no input
    if (valorOriginal) {
        valorInput.val(parseFloat(valorOriginal).toLocaleString('pt-BR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        }));
    }
    
    // Preparar a data antes de mostrar o input
    const dataCell = linha.find('[data-field="data2"]');
    const dataOriginal = dataCell.data('original');
    const dataInput = dataCell.find('.cell-input');
    
    // Garantir que a data esteja no formato correto para o input date (yyyy-mm-dd)
    if (dataOriginal) {
        let dataFormatada = dataOriginal;
        // Se a data está no formato dd/mm/yyyy, converter para yyyy-mm-dd
        if (/^\d{2}\/\d{2}\/\d{4}$/.test(dataOriginal)) {
            const [dia, mes, ano] = dataOriginal.split('/');
            dataFormatada = `${ano}-${mes}-${dia}`;
        }
        dataInput.val(dataFormatada);
    }
    
    linha.find('.cell-display').hide();
    linha.find('.cell-input').show();
    linha.find('.btn-edit, .btn-danger').hide();
    linha.find('.btn-save, .btn-cancel').show();
    
    // Aplicar máscara no campo valor
    FinanceiroClientesModal.aplicarMascaraValorInline(valorInput);
    
    linha.find('input').first().focus();
}

// Função para cancelar edição
FinanceiroClientesModal.cancelarEdicaoInline = function(idPagamento) {
    const linha = $(`tr[data-id="${idPagamento}"]`);
    linha.removeClass('editing-row');
    
    // Restaurar valores originais
    linha.find('.editable-cell').each(function() {
        const campo = $(this).data('field');
        const valorOriginal = $(this).data('original');
        
        if (campo === 'valor') {
            // Formatar o valor para exibição no input
            if (valorOriginal) {
                const valorFormatado = parseFloat(valorOriginal).toLocaleString('pt-BR', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                });
                $(this).find('.cell-input').val(valorFormatado);
            }
        } else if (campo === 'data2') {
            // Garantir que a data esteja no formato correto para o input date
            if (valorOriginal) {
                let dataFormatada = valorOriginal;
                // Se a data está no formato dd/mm/yyyy, converter para yyyy-mm-dd
                if (/^\d{2}\/\d{2}\/\d{4}$/.test(valorOriginal)) {
                    const [dia, mes, ano] = valorOriginal.split('/');
                    dataFormatada = `${ano}-${mes}-${dia}`;
                }
                $(this).find('.cell-input').val(dataFormatada);
            }
        } else {
            $(this).find('.cell-input').val(valorOriginal);
        }
    });
    
    linha.find('.cell-input').hide();
    linha.find('.cell-display').show();
    linha.find('.btn-save, .btn-cancel').hide();
    linha.find('.btn-edit, .btn-danger').show();
}

// Função para salvar edição
FinanceiroClientesModal.salvarEdicaoInline = function(idPagamento) {
    const linha = $(`tr[data-id="${idPagamento}"]`);
    
    const valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    const data = linha.find('[data-field="data2"] .cell-input').val();
    
    const valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    // Debug: log dos valores que serão enviados
    console.log('DEBUG SALVAR EDIÇÃO:', {
        idPagamento: idPagamento,
        valorFormatado: valorFormatado,
        valorParaBanco: valorParaBanco,
        data: data,
        valorNumerico: parseFloat(valorParaBanco)
    });
    
    if (!valorParaBanco || isNaN(parseFloat(valorParaBanco)) || !data) {
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'Todos os campos são obrigatórios e devem ser válidos.' });
        return;
    }
    
    fetch('financeiro_clientes_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'alterar',
            id_financeiro_clientes: idPagamento,
            valor: valorParaBanco,
            data: data,
            id_pedido: FinanceiroClientesModal.ID_PEDIDO
        })
    })
    .then(response => response.json())
    .then(responseData => {
        if (responseData.success) {
            // Atualizar displays
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2, maximumFractionDigits: 2
            }));
            
            // Usar a data original que foi enviada para o servidor
            if (data) {
                const dataFormatada = FinanceiroClientesModal.formatarDataParaExibicao(data);
                linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            }
            
            // Atualizar dados originais
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Sair do modo de edição
            linha.removeClass('editing-row');
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            linha.find('.btn-save, .btn-cancel').hide();
            linha.find('.btn-edit, .btn-danger').show();
            
            FinanceiroClientesModal.showAlert({ icon: 'success', title: 'Sucesso!', text: responseData.message, timer: 1500, showConfirmButton: false });
        } else {
            FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para formatar data para exibição
FinanceiroClientesModal.formatarDataParaExibicao = function(data) {
    if (!data || data === '0000-00-00' || data === '1970-01-01') {
        return 'Sem data';
    }
    
    // Se a data já está no formato dd/mm/yyyy, retorna como está
    if (/^\d{2}\/\d{2}\/\d{4}$/.test(data)) {
        return data;
    }
    
    // Se a data está no formato yyyy-mm-dd, converte para dd/mm/yyyy
    if (/^\d{4}-\d{2}-\d{2}$/.test(data)) {
        try {
            const [ano, mes, dia] = data.split('-');
            return `${dia}/${mes}/${ano}`;
        } catch (error) {
            console.error('Erro ao formatar data:', error);
            return data;
        }
    }
    
    return data;
}

// Funções para edição inline do nome do cliente
FinanceiroClientesModal.editarNomeCliente = function() {
    console.log('✏️ Iniciando edição do nome do cliente...');
    
    const $display = $('.cliente-nome-display');
    const $input = $('.cliente-nome-input');
    const $salvarBtn = $('.cliente-salvar-btn');
    const $cancelarBtn = $('.cliente-cancelar-btn');
    
    // Verificar se já está em modo de edição
    if ($input.is(':visible')) {
        console.log('⚠️ Já está em modo de edição');
        return;
    }
    
    // Mostrar campo de input e botões
    $display.hide();
    $input.show().focus().select();
    $salvarBtn.show();
    $cancelarBtn.show();
    
    console.log('✅ Modo de edição ativado para nome do cliente');
    
    // Adicionar listener para Enter e Escape
    $input.off('keydown.nomeCliente').on('keydown.nomeCliente', function(e) {
        if (e.keyCode === 13) { // Enter
            e.preventDefault();
            FinanceiroClientesModal.salvarNomeCliente();
        } else if (e.keyCode === 27) { // Escape
            e.preventDefault();
            FinanceiroClientesModal.cancelarEdicaoNomeCliente();
        }
    });
};

FinanceiroClientesModal.salvarNomeCliente = function() {
    console.log('💾 Salvando nome do cliente...');
    
    const $display = $('.cliente-nome-display');
    const $input = $('.cliente-nome-input');
    const $salvarBtn = $('.cliente-salvar-btn');
    const $cancelarBtn = $('.cliente-cancelar-btn');
    
    const nomeOriginal = $display.data('original');
    const novoNome = $input.val().trim();
    
    // Validações
    if (!novoNome) {
        FinanceiroClientesModal.showAlert({ 
            icon: 'error', 
            title: 'Erro!', 
            text: 'Nome do cliente é obrigatório.' 
        });
        $input.focus();
        return;
    }
    
    if (novoNome.length < 2) {
        FinanceiroClientesModal.showAlert({ 
            icon: 'error', 
            title: 'Erro!', 
            text: 'Nome deve ter pelo menos 2 caracteres.' 
        });
        $input.focus();
        return;
    }
    
    if (novoNome.length > 100) {
        FinanceiroClientesModal.showAlert({ 
            icon: 'error', 
            title: 'Erro!', 
            text: 'Nome deve ter no máximo 100 caracteres.' 
        });
        $input.focus();
        return;
    }
    
    // Se não houve mudança, apenas cancelar
    if (novoNome === nomeOriginal) {
        console.log('📝 Nenhuma alteração detectada, cancelando edição');
        FinanceiroClientesModal.cancelarEdicaoNomeCliente();
        return;
    }
    
    // Desabilitar botão enquanto salva
    $salvarBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i>');
    
    console.log('🚀 Enviando alteração do nome para o servidor...', {
        nomeOriginal: nomeOriginal,
        novoNome: novoNome,
        idCliente: FinanceiroClientesModal.ID_CLIENTE
    });
    
    fetch('financeiro_clientes_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'alterar_cliente',
            id_cliente: FinanceiroClientesModal.ID_CLIENTE,
            nome_cliente: novoNome
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        return response.json();
    })
    .then(responseData => {
        console.log('📋 Resposta do servidor:', responseData);
        
        if (responseData.success) {
            // Atualizar display com novo nome
            $display.text(responseData.nome_cliente).data('original', responseData.nome_cliente);
            $input.val(responseData.nome_cliente);
            
            // Sair do modo de edição
            $display.show();
            $input.hide();
            $salvarBtn.hide();
            $cancelarBtn.hide();
            
            console.log('✅ Nome do cliente atualizado com sucesso');
            FinanceiroClientesModal.showAlert({ 
                icon: 'success', 
                title: 'Sucesso!', 
                text: responseData.message, 
                timer: 2000, 
                showConfirmButton: false 
            });
            
        } else {
            console.error('❌ Erro do servidor:', responseData.message);
            // Reabilitar botão
            $salvarBtn.prop('disabled', false).html('<i class="fas fa-check"></i>');
            FinanceiroClientesModal.showAlert({ 
                icon: 'error', 
                title: 'Erro!', 
                text: responseData.message 
            });
        }
    })
    .catch(error => {
        console.error('❌ Erro de comunicação:', error);
        // Reabilitar botão
        $salvarBtn.prop('disabled', false).html('<i class="fas fa-check"></i>');
        FinanceiroClientesModal.showAlert({ 
            icon: 'error', 
            title: 'Erro!', 
            text: 'Erro de comunicação com o servidor: ' + error.message 
        });
    });
};

FinanceiroClientesModal.cancelarEdicaoNomeCliente = function() {
    console.log('❌ Cancelando edição do nome do cliente...');
    
    const $display = $('.cliente-nome-display');
    const $input = $('.cliente-nome-input');
    const $salvarBtn = $('.cliente-salvar-btn');
    const $cancelarBtn = $('.cliente-cancelar-btn');
    
    // Restaurar valor original
    const nomeOriginal = $display.data('original');
    $input.val(nomeOriginal);
    
    // Sair do modo de edição
    $display.show();
    $input.hide();
    $salvarBtn.hide();
    $cancelarBtn.hide();
    
    // Remover listeners
    $input.off('keydown.nomeCliente');
    
    console.log('✅ Edição do nome cancelada');
};

// Função para adicionar nova linha
FinanceiroClientesModal.adicionarNovaLinha = function() {
    console.log('🆕 Iniciando adição de nova linha...');
    
    // Cancelar outras edições em andamento
    FinanceiroClientesModal.cancelarTodasEdicoes();
    
    // Ocultar mensagem de "nenhum pagamento" se existir
    $('#mensagem-vazio').hide();
    
    const novoId = 'novo_' + Date.now();
    console.log('🔄 ID da nova linha:', novoId);
    
    // Verificar se DataTable está ativo
    if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
        console.log('📊 Usando DataTable API para adicionar linha...');
        const table = $('#tabelaPagamentos').DataTable();
        
        // Adicionar nova linha usando a API do DataTable
        const novaLinhaData = [
            'Novo',
            'R$ 0,00',
            new Date().toLocaleDateString('pt-BR'),
            '' // Ações serão adicionadas após
        ];
        
        const novaLinha = table.row.add(novaLinhaData).draw(false);
        const $novaLinhaElement = $(novaLinha.node());
        
        // Configurar a linha como nova linha editável
        $novaLinhaElement.attr('data-id', novoId);
        $novaLinhaElement.addClass('nova-linha editing-row');
        
        // Substituir conteúdo das células para modo de edição
        $novaLinhaElement.html(`
            <td>Novo</td>
            <td class="editable-cell" data-field="valor" data-original="">
                <span class="cell-display" style="display: none;">R$ 0,00</span>
                <input type="text" class="cell-input form-control form-control-sm valor-input" value="" placeholder="0,00">
            </td>
            <td class="editable-cell" data-field="data2" data-original="">
                <span class="cell-display" style="display: none;">Sem data</span>
                <input type="date" class="cell-input form-control form-control-sm" value="${new Date().toISOString().split('T')[0]}">
            </td>
            <td>
                <button type="button" class="btn btn-sm btn-success btn-save" onclick="FinanceiroClientesModal.salvarNovaLinha('${novoId}')" title="Salvar">
                    <i class="fas fa-check"></i>
                </button>
                <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="FinanceiroClientesModal.cancelarNovaLinha('${novoId}')" title="Cancelar">
                    <i class="fas fa-times"></i>
                </button>
            </td>
        `);
        
        // Aplicar máscara no novo campo valor
        FinanceiroClientesModal.aplicarMascaraValorInline($novaLinhaElement.find('.valor-input'));
        
        // Focar no primeiro campo
        $novaLinhaElement.find('.valor-input').focus();
        
        console.log('✅ Nova linha adicionada via DataTable API');
        
    } else {
        console.log('📋 Usando DOM direto para adicionar linha...');
        
        // Fallback: adicionar diretamente ao DOM se DataTable não estiver ativo
        const novaLinha = `
            <tr data-id="${novoId}" class="nova-linha editing-row">
                <td>Novo</td>
                <td class="editable-cell" data-field="valor" data-original="">
                    <span class="cell-display" style="display: none;">R$ 0,00</span>
                    <input type="text" class="cell-input form-control form-control-sm valor-input" value="" placeholder="0,00">
                </td>
                <td class="editable-cell" data-field="data2" data-original="">
                    <span class="cell-display" style="display: none;">Sem data</span>
                    <input type="date" class="cell-input form-control form-control-sm" value="${new Date().toISOString().split('T')[0]}">
                </td>
                <td>
                    <button type="button" class="btn btn-sm btn-success btn-save" onclick="FinanceiroClientesModal.salvarNovaLinha('${novoId}')" title="Salvar">
                        <i class="fas fa-check"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="FinanceiroClientesModal.cancelarNovaLinha('${novoId}')" title="Cancelar">
                        <i class="fas fa-times"></i>
                    </button>
                </td>
            </tr>
        `;
        
        $('#tabelaPagamentos tbody').prepend(novaLinha);
        
        // Aplicar máscara no novo campo valor
        const novaLinhaElement = $(`tr[data-id="${novoId}"]`);
        FinanceiroClientesModal.aplicarMascaraValorInline(novaLinhaElement.find('.valor-input'));
        
        // Focar no primeiro campo
        novaLinhaElement.find('.valor-input').focus();
        
        console.log('✅ Nova linha adicionada via DOM direto');
    }
    
    // Verificação adicional de persistência
    console.log('🔍 Verificando se nova linha persiste...');
    setTimeout(function() {
        const linhaVerificacao = $(`tr[data-id="${novoId}"]`);
        if (linhaVerificacao.length > 0) {
            console.log('✅ Nova linha confirmada presente na tabela');
            console.log('📊 Classes da linha:', linhaVerificacao.attr('class'));
            console.log('📍 Posição da linha:', linhaVerificacao.index());
        } else {
            console.log('❌ ALERTA: Nova linha desapareceu após criação!');
            console.log('🔍 Investigando tabela atual...');
            console.log('📈 Total de linhas na tabela:', $('#tabelaPagamentos tbody tr').length);
        }
    }, 100);
}

// Função para salvar nova linha
FinanceiroClientesModal.salvarNovaLinha = function(novoId) {
    console.log('🚀 INICIANDO salvarNovaLinha para ID:', novoId);
    
    // Verificar se já está processando para evitar cliques duplos
    if (FinanceiroClientesModal.processandoSalvar) {
        console.log('⏳ JÁ ESTÁ PROCESSANDO, ignorando clique');
        return;
    }
    
    FinanceiroClientesModal.processandoSalvar = true;
    
    const linha = $(`tr[data-id="${novoId}"]`);
    
    if (linha.length === 0) {
        console.error('❌ Linha não encontrada para ID:', novoId);
        FinanceiroClientesModal.processandoSalvar = false;
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'Linha não encontrada.' });
        return;
    }
    
    const valorInput = linha.find('[data-field="valor"] .cell-input');
    const dataInput = linha.find('[data-field="data2"] .cell-input');
    
    const valorFormatado = valorInput.val();
    const data = dataInput.val();
    
    console.log('📊 VALORES CAPTURADOS:', {
        valorFormatado: valorFormatado,
        data: data,
        valorLength: valorFormatado ? valorFormatado.length : 0,
        dataLength: data ? data.length : 0,
        inputsEncontrados: {
            valorInput: valorInput.length,
            dataInput: dataInput.length
        }
    });
    
    if (!valorFormatado || valorFormatado.trim() === '') {
        console.error('❌ VALOR VAZIO');
        FinanceiroClientesModal.processandoSalvar = false;
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'O valor é obrigatório.' });
        valorInput.focus();
        return;
    }
    
    if (!data || data.trim() === '') {
        console.error('❌ DATA VAZIA');
        FinanceiroClientesModal.processandoSalvar = false;
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'A data é obrigatória.' });
        dataInput.focus();
        return;
    }
    
    const valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    console.log('💾 VALOR PARA BANCO:', {
        valorParaBanco: valorParaBanco,
        valorNumerico: parseFloat(valorParaBanco),
        isNaN: isNaN(parseFloat(valorParaBanco)),
        valorMaiorQueZero: parseFloat(valorParaBanco) > 0
    });
    
    if (!valorParaBanco || isNaN(parseFloat(valorParaBanco)) || parseFloat(valorParaBanco) <= 0) {
        console.error('❌ VALIDAÇÃO DE VALOR FALHOU');
        FinanceiroClientesModal.processandoSalvar = false;
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'O valor deve ser um número maior que zero.' });
        valorInput.focus();
        return;
    }
    
    console.log('✅ VALIDAÇÃO OK, enviando para servidor...');
    
    // Desabilitar botão para evitar cliques múltiplos
    const btnSalvar = linha.find('.btn-save');
    btnSalvar.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i>');
    
    fetch('financeiro_clientes_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'inserir',
            valor: valorParaBanco,
            data: data,
            id_pedido: FinanceiroClientesModal.ID_PEDIDO
        })
    })
    .then(response => {
        console.log('📡 RESPOSTA RECEBIDA:', response.status, response.statusText);
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        return response.json();
    })
    .then(responseData => {
        console.log('📋 DADOS DA RESPOSTA:', responseData);
        
        if (responseData.success) {
            console.log('✅ PAGAMENTO SALVO COM SUCESSO!');
            
            // Ocultar mensagem de "nenhum pagamento" se existir
            $('#mensagem-vazio').hide();
            
            // Atualizar a linha com o ID real
            linha.attr('data-id', responseData.id_financeiro_clientes);
            linha.removeClass('nova-linha editing-row');
            
            // Atualizar primeira coluna com ID real
            linha.find('td:first').text(responseData.id_financeiro_clientes);
            
            console.log('🆔 Linha atualizada com ID real:', responseData.id_financeiro_clientes);
            
            // Atualizar displays
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2, maximumFractionDigits: 2
            }));
            
            const dataFormatada = FinanceiroClientesModal.formatarDataParaExibicao(data);
            linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            
            // Atualizar dados originais
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Mostrar displays e esconder inputs
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            
            // Trocar botões
            linha.find('.btn-save, .btn-cancel').remove();
            linha.find('td:last').html(`
                <button type="button" class="btn btn-sm btn-warning btn-edit" onclick="FinanceiroClientesModal.editarInline(${responseData.id_financeiro_clientes})" title="Editar">
                    <i class="fas fa-edit"></i>
                </button>
                <button type="button" class="btn btn-sm btn-danger" onclick="FinanceiroClientesModal.excluirPagamento(${responseData.id_financeiro_clientes})" title="Excluir">
                    <i class="fas fa-trash"></i>
                </button>
            `);
            
            console.log('✅ PAGAMENTO SALVO COM SUCESSO!');
            FinanceiroClientesModal.processandoSalvar = false;
            
            // Forçar atualização e reordenamento do DataTable
            if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
                const table = $('#tabelaPagamentos').DataTable();
                
                // Primeiro, invalidar os dados para forçar o DataTable a reconhecer as mudanças
                table.rows().invalidate('data');
                
                // Depois aplicar ordenação e redraw
                table.order([[ 0, 'desc' ]]).draw(false);
                
                console.log('🔄 DataTable atualizado e reordenado para mostrar últimos registros primeiro');
                
                // Garantir que a linha recém-criada esteja visível no topo
                setTimeout(function() {
                    // Verificar se a linha existe e está visível
                    const $linhaAtualizada = $(`tr[data-id="${responseData.id_financeiro_clientes}"]`);
                    if ($linhaAtualizada.length > 0) {
                        console.log('✅ Linha encontrada na tabela após salvamento');
                        
                        // Forçar scroll para o topo da tabela se necessário
                        const tableContainer = $('#tabelaPagamentos_wrapper');
                        if (tableContainer.length > 0) {
                            tableContainer[0].scrollIntoView({ behavior: 'smooth', block: 'start' });
                        }
                        
                        // Destacar brevemente a linha recém-salva
                        $linhaAtualizada.addClass('table-success');
                        setTimeout(function() {
                            $linhaAtualizada.removeClass('table-success');
                        }, 2000);
                        
                    } else {
                        console.log('⚠️ Linha não encontrada após salvamento, verificando...');
                        // Se não encontrou, forçar redraw mais agressivo
                        table.draw(true);
                    }
                }, 200);
            } else {
                console.log('⚠️ DataTable não encontrado, linha salva mas pode não estar visível');
            }
            
            FinanceiroClientesModal.showAlert({ icon: 'success', title: 'Sucesso!', text: responseData.message, timer: 1500, showConfirmButton: false });
            
            // Verificação final após salvamento
            setTimeout(function() {
                console.log('🔍 Verificação final após salvamento...');
                const linhaFinal = $(`tr[data-id="${responseData.id_financeiro_clientes}"]`);
                const totalLinhas = $('#tabelaPagamentos tbody tr').length;
                
                console.log('📊 Status final:', {
                    linhaEncontrada: linhaFinal.length > 0,
                    totalLinhas: totalLinhas,
                    idProcurado: responseData.id_financeiro_clientes,
                    posicaoLinha: linhaFinal.length > 0 ? linhaFinal.index() : 'N/A'
                });
                
                if (linhaFinal.length === 0) {
                    console.log('❌ CRÍTICO: Linha desapareceu após salvamento!');
                    console.log('🔍 Todas as linhas na tabela:');
                    $('#tabelaPagamentos tbody tr').each(function(index) {
                        const id = $(this).data('id') || $(this).attr('data-id');
                        console.log(`  ${index}: ID = ${id}`);
                    });
                } else {
                    console.log('✅ Linha salva e visível corretamente');
                }
            }, 500);
            
            // Marcar como processamento concluído
            FinanceiroClientesModal.processandoSalvar = false;
            
            console.log('🎉 SUCESSO: Novo pagamento salvo e processamento concluído');
        } else {
            console.error('❌ ERRO DO SERVIDOR:', responseData.message);
            // Reabilitar botão em caso de erro
            btnSalvar.prop('disabled', false).html('<i class="fas fa-check"></i>');
            FinanceiroClientesModal.processandoSalvar = false;
            FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: responseData.message || 'Erro desconhecido no servidor.' });
        }
    })
    .catch(error => {
        console.error('❌ ERRO DE COMUNICAÇÃO:', error);
        // Reabilitar botão em caso de erro
        btnSalvar.prop('disabled', false).html('<i class="fas fa-check"></i>');
        FinanceiroClientesModal.processandoSalvar = false;
        FinanceiroClientesModal.showAlert({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor: ' + error.message });
    });
}

// Função para cancelar nova linha
FinanceiroClientesModal.cancelarNovaLinha = function(novoId) {
    console.log('🗑️ Cancelando nova linha:', novoId);
    
    // Verificar se a linha ainda está em modo de edição
    const linha = $(`tr[data-id="${novoId}"]`);
    if (linha.length > 0 && linha.hasClass('nova-linha')) {
        console.log('✅ Linha cancelada e removida');
        
        // Se DataTable estiver ativo, usar API para remover
        if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
            const table = $('#tabelaPagamentos').DataTable();
            const rowToRemove = table.row(linha);
            if (rowToRemove.length > 0) {
                rowToRemove.remove().draw(false);
                console.log('🗑️ Linha removida via DataTable API');
            } else {
                linha.remove();
                console.log('🗑️ Linha removida via DOM direto');
            }
        } else {
            linha.remove();
            console.log('🗑️ Linha removida via DOM direto');
        }
    } else {
        console.log('⚠️ Linha não encontrada ou não está em edição:', novoId);
    }
}

// Função para excluir pagamento
FinanceiroClientesModal.excluirPagamento = function(idPagamento) {
    Swal.fire({
        title: 'Tem certeza?',
        text: 'Esta ação não pode ser desfeita!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sim, excluir!',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch('financeiro_clientes_modal.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    acao: 'excluir',
                    id_financeiro_clientes: idPagamento,
                    id_pedido: FinanceiroClientesModal.ID_PEDIDO
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    $(`tr[data-id="${idPagamento}"]`).remove();
                    Swal.fire({ icon: 'success', title: 'Excluído!', text: data.message, timer: 1500, showConfirmButton: false });
                } else {
                    Swal.fire({ icon: 'error', title: 'Erro!', text: data.message });
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
            });
        }
    });
}

// Função para preservar dados da tabela durante reinicializações
FinanceiroClientesModal.preservarDadosTabela = function() {
    console.log('💾 Preservando dados da tabela...');
    
    const $tabela = $('#tabelaPagamentos tbody');
    if ($tabela.length > 0) {
        // Salvar HTML atual da tabela
        FinanceiroClientesModal.dadosBackup = $tabela.html();
        const numLinhas = $tabela.find('tr').length;
        console.log(`✅ Backup criado: ${numLinhas} linhas salvas`);
        return true;
    }
    return false;
};

// Função para restaurar dados da tabela se necessário
FinanceiroClientesModal.restaurarDadosTabela = function() {
    console.log('🔄 Verificando se precisa restaurar dados...');
    
    const $tabela = $('#tabelaPagamentos tbody');
    const numLinhasAtual = $tabela.find('tr').length;
    
    if (numLinhasAtual === 0 && FinanceiroClientesModal.dadosBackup) {
        console.log('🚨 Tabela vazia detectada, restaurando backup...');
        $tabela.html(FinanceiroClientesModal.dadosBackup);
        const numLinhasRestauradas = $tabela.find('tr').length;
        console.log(`✅ Dados restaurados: ${numLinhasRestauradas} linhas`);
        return true;
    }
    
    console.log(`📊 Tabela OK: ${numLinhasAtual} linhas presentes`);
    return false;
};

// Função para garantir persistência das informações do pedido
FinanceiroClientesModal.garantirInfoPedidoPersistente = function() {
    console.log('🔒 Ativando proteção persistente das informações do pedido...');
    
    // Função para verificar e restaurar
    function verificarERestaurar() {
        const $container = $('#info-pedido-container');
        const $alert = $('#info-pedido-alert');
        
        if ($container.length === 0) {
            console.log('❌ Container de informações perdido! Tentando restaurar...');
            // Se perdido, tentar encontrar e recriar baseado nos dados PHP
            const infoPedidoData = {
                pedidoId: FinanceiroClientesModal.ID_PEDIDO,
                clienteId: FinanceiroClientesModal.ID_CLIENTE
            };
            
            // Recriar o container básico se necessário
            if ($('.container-fluid .row:first').length === 0 || !$('.container-fluid .row:first').hasClass('mb-3')) {
                const infoHTML = `
                    <div class="row mb-3" id="info-pedido-container">
                        <div class="col-12">
                            <div class="alert alert-info" id="info-pedido-alert">
                                <strong>Pedido ID:</strong> ${infoPedidoData.pedidoId} | 
                                <strong>Cliente ID:</strong> ${infoPedidoData.clienteId}
                            </div>
                        </div>
                    </div>
                `;
                $('.container-fluid').prepend(infoHTML);
            }
        }
        
        // Forçar visibilidade
        if ($container.length > 0) {
            $container[0].style.setProperty('display', 'block', 'important');
            $container[0].style.setProperty('visibility', 'visible', 'important');
            $container[0].style.setProperty('opacity', '1', 'important');
        }
        
        if ($alert.length > 0) {
            $alert[0].style.setProperty('display', 'block', 'important');
            $alert[0].style.setProperty('visibility', 'visible', 'important');
            $alert[0].style.setProperty('opacity', '1', 'important');
        }
    }
    
    // Verificação contínua
    verificarERestaurar();
    setTimeout(verificarERestaurar, 100);
    setTimeout(verificarERestaurar, 300);
    setTimeout(verificarERestaurar, 600);
    
    // Observer contínuo
    if (typeof MutationObserver !== 'undefined') {
        if (FinanceiroClientesModal.persistenceObserver) {
            FinanceiroClientesModal.persistenceObserver.disconnect();
        }
        
        FinanceiroClientesModal.persistenceObserver = new MutationObserver(function(mutations) {
            let shouldProtect = false;
            mutations.forEach(function(mutation) {
                if (mutation.type === 'attributes' || mutation.type === 'childList') {
                    // Verificar se as informações do pedido foram afetadas
                    const target = mutation.target;
                    if (target && (target.id === 'info-pedido-container' || target.id === 'info-pedido-alert' || 
                        target.classList && (target.classList.contains('alert') || target.classList.contains('alert-info')))) {
                        shouldProtect = true;
                    }
                }
            });
            
            if (shouldProtect) {
                setTimeout(verificarERestaurar, 10);
            }
        });
        
        // Observar mudanças no documento
        const container = document.querySelector('.container-fluid');
        if (container) {
            FinanceiroClientesModal.persistenceObserver.observe(container, {
                childList: true,
                subtree: true,
                attributes: true,
                attributeFilter: ['style', 'class', 'hidden']
            });
        }
    }
};

// Inicialização
$(document).ready(function() {
    console.log('🔄 Document ready executado...');
    
    // PRIMEIRA COISA: Ativar proteção persistente das informações
    FinanceiroClientesModal.garantirInfoPedidoPersistente();
    
    // Sempre limpar estado anterior primeiro
    if (window.financeiroModalCarregado) {
        console.log('🧹 Limpando estado anterior...');
        
        // Destruir DataTable existente
        if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
            try {
                $('#tabelaPagamentos').DataTable().destroy(false);
                console.log('✅ DataTable anterior destruído preservando dados');
            } catch (e) {
                console.log('⚠️ Erro ao destruir DataTable anterior:', e);
            }
        }
        
        // Limpar wrappers residuais
        $('.dataTables_wrapper').remove();
        $('#tabelaPagamentos').removeClass('dataTable');
        $('#tabelaPagamentos').removeAttr('role aria-describedby');
        
        // Limpar timeouts
        for (let i = 0; i < 10; i++) {
            clearTimeout(window['timeout_financeiro_col_' + i]);
        }
        
        // Aguardar limpeza
        setTimeout(function() {
            inicializarModalFinanceiro();
        }, 200);
    } else {
        // Primeira execução
        inicializarModalFinanceiro();
    }
});

// Função separada para inicialização
function inicializarModalFinanceiro() {
    console.log('� Inicializando modal financeiro...');
    
    // Marcar como carregado
    window.financeiroModalCarregado = true;
    
    // Garantir que as informações do pedido permaneçam visíveis
    FinanceiroClientesModal.protegerInfoPedido();
    
    // DEBUG: Verificar estado inicial da tabela
    const $tabela = $('#tabelaPagamentos');
    const numLinhasInicial = $tabela.find('tbody tr').length;
    console.log('📊 Estado inicial após proteção - Linhas na tabela:', numLinhasInicial);
    
    // Preservar dados antes de qualquer operação
    FinanceiroClientesModal.preservarDadosTabela();
    
    if (numLinhasInicial > 0) {
        console.log('✅ Dados da tabela preservados');
        $tabela.find('tbody tr').slice(0, 2).each(function(index) {
            const id = $(this).find('td:first').text();
            const valor = $(this).find('td:nth-child(2)').text();
            console.log(`📋 Linha ${index + 1}: ID=${id}, Valor=${valor}`);
        });
    } else {
        console.log('❌ PROBLEMA: Tabela ficou vazia!');
    }
    
    // Aplicar máscara nos campos de valor existentes
    $('.valor-input').each(function() {
        FinanceiroClientesModal.aplicarMascaraValorInline($(this));
    });
    
    // Inicializar DataTable com filtros por coluna
    if ($.fn.DataTable && $('#tabelaPagamentos').length > 0) {
        console.log('🔍 Verificando estado do DataTable...');
        
        // Limpar qualquer instância anterior de forma mais agressiva
        if ($.fn.DataTable.isDataTable('#tabelaPagamentos')) {
            console.log('�️ Destruindo DataTable existente...');
            try {
                $('#tabelaPagamentos').DataTable().destroy(false);
                // NÃO limpar o conteúdo HTML da tabela para preservar estrutura
                console.log('✅ DataTable destruído preservando estrutura HTML');
            } catch (e) {
                console.log('⚠️ Erro ao destruir DataTable:', e);
            }
        }
        
        // Remover qualquer wrapper restante
        $('.dataTables_wrapper').remove();
        
        // Aguardar um momento antes de recriar
        setTimeout(function() {
            console.log('🚀 Inicializando novo DataTable...');
            console.log('📋 Estado da tabela antes da inicialização:');
            console.log('  - Tabela existe:', $('#tabelaPagamentos').length > 0);
            console.log('  - Thead existe:', $('#tabelaPagamentos thead').length > 0);  
            console.log('  - Tbody existe:', $('#tabelaPagamentos tbody').length > 0);
            console.log('  - Linhas no tbody:', $('#tabelaPagamentos tbody tr').length);
            
            try {
                var table = $('#tabelaPagamentos').DataTable({
                    "language": {
                        "sEmptyTable": "Nenhum registro encontrado",
                        "sInfo": "Mostrando de _START_ até _END_ de _TOTAL_ registros",
                        "sInfoEmpty": "Mostrando 0 até 0 de 0 registros",
                        "sInfoFiltered": "(Filtrados de _MAX_ registros)",
                        "sInfoPostFix": "",
                        "sInfoThousands": ".",
                        "sLengthMenu": "_MENU_ resultados por página",
                        "sLoadingRecords": "Carregando...",
                        "sProcessing": "Processando...",
                        "sZeroRecords": "Nenhum registro encontrado",
                        "sSearch": "Pesquisar",
                        "oPaginate": {
                            "sNext": "Próximo",
                            "sPrevious": "Anterior",
                            "sFirst": "Primeiro",
                            "sLast": "Último"
                        },
                        "oAria": {
                            "sSortAscending": ": Ordenar colunas de forma ascendente",
                            "sSortDescending": ": Ordenar colunas de forma descendente"
                        }
                    },
                    "pageLength": 25,
                    "lengthMenu": [ [10, 25, 50, -1], [10, 25, 50, "Todos"] ],
                    "dom": 'lfrtip', // Usar dom simples para evitar conflitos
                    "responsive": true,
                    "processing": false,
                    "info": true,
                    "paging": true,
                    "searching": true,
                    "ordering": true,
                    "order": [[ 0, "desc" ]], // Ordenar pela primeira coluna (ID) em ordem decrescente para mostrar últimos registros primeiro
                    "search": {
                        "regex": false,
                        "caseInsensitive": true
                    },
                    "initComplete": function () {
                        console.log('✅ DataTable Financeiro inicializado com sucesso');
                        
                        // DEBUG: Verificar se os dados ainda estão na tabela após inicialização
                        const numLinhasAposInit = $('#tabelaPagamentos tbody tr').length;
                        console.log('📊 Após inicialização DataTable - Linhas:', numLinhasAposInit);
                        
                        if (numLinhasAposInit === 0) {
                            console.log('❌ PROBLEMA CRÍTICO: Dados foram perdidos durante inicialização do DataTable!');
                            // Tentar restaurar dados do backup
                            if (FinanceiroClientesModal.restaurarDadosTabela()) {
                                console.log('🔄 Dados restaurados, reinicializando DataTable...');
                                // Reinicializar DataTable com os dados restaurados
                                setTimeout(function() {
                                    if ($.fn.DataTable.isDataTable('#tabelaPagamentos')) {
                                        $('#tabelaPagamentos').DataTable().destroy(false);
                                    }
                                    $('#tabelaPagamentos').DataTable({
                                        "order": [[ 0, "desc" ]],
                                        "pageLength": 25
                                    });
                                }, 100);
                            }
                        } else {
                            console.log('✅ Dados preservados durante inicialização');
                        }
                        
                        // Proteger informações do pedido após inicialização
                        FinanceiroClientesModal.protegerInfoPedido();
                        
                        // Esconder busca global após inicialização
                        $('#tabelaPagamentos_filter').hide();
                        
                        // Garantir ordenação correta (últimos registros primeiro)
                        this.api().order([[ 0, 'desc' ]]).draw(false);
                        console.log('🔄 Ordem inicial configurada: últimos registros primeiro');
                        
                        setupFooterSearchFinanceiro();
                    },
                    "drawCallback": function (settings) {
                        // Proteger informações do pedido após cada redraw
                        FinanceiroClientesModal.protegerInfoPedido();
                        
                        // Garantir que a busca global permanece oculta
                        $('#tabelaPagamentos_filter').hide();
                        
                        // Reinicializar tooltips nas novas linhas
                        $('[data-toggle="tooltip"]').tooltip('dispose').tooltip({
                            delay: { show: 300, hide: 100 }
                        });
                        
                        // Verificar se os inputs ainda existem após redraw
                        var $inputs = $('.footer-search-input-financeiro');
                        if ($inputs.length === 0) {
                            console.log('⚠️ DRAW CALLBACK: Inputs perdidos após redraw, recriando...');
                            setTimeout(function() {
                                criarInputsFinanceiroManualmente();
                            }, 200);
                        }
                    }
                });
                
                console.log('🎯 DataTable criado com sucesso');
                
            } catch (error) {
                console.error('❌ Erro ao criar DataTable:', error);
            }
        }, 200); // Fechar o setTimeout da inicialização
        
        // Função para configurar busca no footer
        function setupFooterSearchFinanceiro() {
            setTimeout(function() {
                console.log('🔧 SETUP FOOTER SEARCH FINANCEIRO - Iniciando...');
                
                // Verificar se já existem inputs criados
                var $inputsExistentes = $('.footer-search-input-financeiro');
                if ($inputsExistentes.length > 0) {
                    console.log('✅ Inputs já criados, pulando criação');
                    return;
                }
                
                // Garantir que temos referência da tabela
                if (!table || !table.columns) {
                    console.log('❌ Tabela não encontrada, tentando obter referência...');
                    if ($.fn.DataTable.isDataTable('#tabelaPagamentos')) {
                        table = $('#tabelaPagamentos').DataTable();
                    } else {
                        console.log('❌ DataTable não está inicializado');
                        return;
                    }
                }
                
                // Tentar método automático primeiro
                var inputsCriados = 0;
                try {
                    table.columns().every(function (index) {
                        var column = this;
                        var $footerCell = $(column.footer());
                        var title = $footerCell.text().trim();
                        
                        console.log('📋 Processando coluna', index, 'título:', title);
                        
                        // Pular coluna de Ações (última coluna - índice 3)
                        if (index === 3) {
                            $footerCell.html('');
                            console.log('⏭️ Pulando coluna de ações (índice 3)');
                            return;
                        }
                        
                        if (title) {
                            try {
                                // Limpar conteúdo anterior
                                $footerCell.empty();
                                
                                // Criar input com classes padronizadas (mesmo estilo do pedidos.php)
                                var inputId = 'footer-input-col-' + index;
                                var inputHtml = '<div class="md-input-wrapper">' +
                                    '<input type="text" id="' + inputId + '" class="md-form-control footer-search-input-financeiro" ' +
                                    'placeholder="Filtrar ' + title.replace(/'/g, '&#39;').replace(/"/g, '&quot;') + '" ' +
                                    'data-column="' + index + '" />' +
                                    '</div>';
                                
                                $footerCell.html(inputHtml);
                                inputsCriados++;
                                
                                console.log('✅ Input criado para coluna', index);
                                
                                // Configurar evento de busca com debounce melhorado
                                var $input = $footerCell.find('.footer-search-input-financeiro');
                                var searchTimeout;
                                
                                $input.on('keyup change clear paste input', function () {
                                    var searchValue = this.value.trim();
                                    var columnIndex = parseInt($(this).data('column'));
                                    
                                    console.log('🔍 Filtro aplicado - Coluna:', columnIndex, 'Valor:', searchValue);
                                    
                                    // Limpar timeout anterior
                                    clearTimeout(searchTimeout);
                                    
                                    // Aplicar filtro com delay
                                    searchTimeout = setTimeout(function() {
                                        try {
                                            var currentSearch = column.search();
                                            if (currentSearch !== searchValue) {
                                                if (searchValue === '') {
                                                    // Limpar filtro se vazio
                                                    column.search('', false, false);
                                                } else {
                                                    // Aplicar filtro - usar busca simples sem regex
                                                    column.search(searchValue, false, false);
                                                }
                                                
                                                // Redesenhar tabela
                                                column.draw();
                                                
                                                console.log('✅ Filtro aplicado com sucesso na coluna', columnIndex);
                                            }
                                        } catch (error) {
                                            console.error('❌ Erro ao aplicar filtro na coluna', columnIndex, ':', error);
                                        }
                                    }, 300); // Delay de 300ms
                                });
                                
                                // Prevenir ordenação ao clicar no input
                                $input.on('click focus', function(e) {
                                    e.stopPropagation();
                                });
                                
                                // Prevenir submit do form ao pressionar Enter
                                $input.on('keydown', function(e) {
                                    if (e.keyCode === 13) { // Enter
                                        e.preventDefault();
                                        e.stopPropagation();
                                        return false;
                                    }
                                });
                                
                            } catch (error) {
                                console.log('❌ Erro ao criar input para coluna', index, ':', error);
                            }
                        }
                    });
                    
                    console.log('📊 Total de inputs criados pelo método automático:', inputsCriados);
                    
                } catch (error) {
                    console.log('❌ Erro no método automático:', error);
                    inputsCriados = 0;
                }
                
                // Se não conseguiu criar automaticamente, forçar criação manual
                if (inputsCriados === 0) {
                    console.log('⚠️ Método automático falhou, forçando criação manual...');
                    criarInputsFinanceiroManualmente();
                } else {
                    console.log('✅ Setup concluído com sucesso! Inputs criados:', inputsCriados);
                    
                    // Verificar se os inputs realmente existem e estão funcionando
                    setTimeout(function() {
                        var $verificarInputs = $('.footer-search-input-financeiro');
                        console.log('🔍 Verificação final - Inputs encontrados:', $verificarInputs.length);
                        
                        $verificarInputs.each(function(index) {
                            var coluna = $(this).data('column');
                            var placeholder = $(this).attr('placeholder');
                            console.log('  Input', index + 1, '- Coluna:', coluna, 'Placeholder:', placeholder);
                        });
                        
                        if ($verificarInputs.length === 0) {
                            console.log('❌ NENHUM INPUT ENCONTRADO após criação, forçando criação manual...');
                            criarInputsFinanceiroManualmente();
                        }
                    }, 500);
                }
            }, 100); // Reduzir delay inicial
        }
        
        // Função para criar inputs manualmente - MELHORADA
        function criarInputsFinanceiroManualmente() {
            console.log('🔧 CRIAÇÃO MANUAL FINANCEIRO - Iniciando...');
            
            // Definir cabeçalhos das colunas com base na estrutura HTML
            var colunas = [
                { titulo: 'Nº', placeholder: 'Filtrar por Número' },
                { titulo: 'Valor', placeholder: 'Filtrar por Valor' },
                { titulo: 'Data', placeholder: 'Filtrar por Data' },
                { titulo: 'Ações', placeholder: '' }
            ];
            
            // Verificar se footer existe, se não, criar
            var $footer = $('#tabelaPagamentos tfoot');
            if ($footer.length === 0) {
                console.log('❌ Footer não encontrado, criando...');
                var footerHtml = '<tfoot><tr>';
                for (var i = 0; i < colunas.length; i++) {
                    footerHtml += '<th>' + colunas[i].titulo + '</th>';
                }
                footerHtml += '</tr></tfoot>';
                $('#tabelaPagamentos').append(footerHtml);
                $footer = $('#tabelaPagamentos tfoot');
                console.log('✅ Footer criado com ' + colunas.length + ' colunas');
            }
            
            var $footerCells = $footer.find('th');
            console.log('📋 Footer cells encontradas:', $footerCells.length);
            
            // Obter referência da tabela DataTable
            var tableRef = null;
            if ($.fn.DataTable.isDataTable('#tabelaPagamentos')) {
                tableRef = $('#tabelaPagamentos').DataTable();
            }
            
            if (!tableRef) {
                console.log('❌ Referência da tabela não encontrada');
                return;
            }
            
            // Criar inputs para cada coluna (exceto a última - Ações)
            for (var i = 0; i < colunas.length - 1; i++) {
                var coluna = colunas[i];
                var $cell = $footerCells.eq(i);
                
                if ($cell.length > 0) {
                    console.log('🔧 Criando input para coluna', i, ':', coluna.titulo);
                    
                    // Limpar conteúdo anterior
                    $cell.empty();
                    
                    // Criar input com wrapper e classes padronizadas (mesmo estilo do pedidos.php)
                    var inputId = 'footer-manual-input-col-' + i;
                    var inputHtml = '<div class="md-input-wrapper">' +
                        '<input type="text" id="' + inputId + '" class="md-form-control footer-search-input-financeiro" ' +
                        'placeholder="' + coluna.placeholder + '" data-column="' + i + '" />' +
                        '</div>';
                    
                    $cell.html(inputHtml);
                    
                    // Configurar evento de busca com escopo correto
                    var $input = $cell.find('.footer-search-input-financeiro');
                    
                    // Usar closure para manter o contexto da coluna
                    (function(columnIndex, tableReference) {
                        $input.on('keyup change clear paste input', function() {
                            var searchValue = $(this).val().trim();
                            
                            console.log('🔍 Filtro manual aplicado - Coluna:', columnIndex, 'Valor:', searchValue);
                            
                            // Aplicar filtro na coluna com throttling
                            clearTimeout(window['timeout_financeiro_manual_col_' + columnIndex]);
                            window['timeout_financeiro_manual_col_' + columnIndex] = setTimeout(function() {
                                try {
                                    var column = tableReference.column(columnIndex);
                                    if (column && typeof column.search === 'function') {
                                        var currentSearch = column.search();
                                        if (currentSearch !== searchValue) {
                                            if (searchValue === '') {
                                                column.search('', false, false);
                                            } else {
                                                column.search(searchValue, false, false);
                                            }
                                            column.draw();
                                            console.log('✅ Filtro manual aplicado na coluna', columnIndex);
                                        }
                                    } else {
                                        console.error('❌ Coluna', columnIndex, 'não encontrada ou método search não disponível');
                                    }
                                } catch (error) {
                                    console.error('❌ Erro ao aplicar filtro manual na coluna', columnIndex, ':', error);
                                }
                            }, 300);
                        });
                        
                        // Prevenir ordenação ao clicar no input
                        $input.on('click focus', function(e) {
                            e.stopPropagation();
                        });
                        
                        // Prevenir submit ao pressionar Enter
                        $input.on('keydown', function(e) {
                            if (e.keyCode === 13) {
                                e.preventDefault();
                                e.stopPropagation();
                                return false;
                            }
                        });
                    })(i, tableRef);
                    
                    console.log('✅ Input manual criado para coluna', i);
                } else {
                    console.log('❌ Cell não encontrada para coluna', i);
                }
            }
            
            // Limpar última coluna (Ações)
            if ($footerCells.length > 3) {
                $footerCells.eq(3).empty();
                console.log('🧹 Coluna de ações limpa');
            }
            
            console.log('🔧 CRIAÇÃO MANUAL FINANCEIRO - Concluída!');
            
            // Verificação final
            setTimeout(function() {
                var $inputsFinais = $('.footer-search-input-financeiro');
                console.log('🔍 Verificação final manual - Inputs criados:', $inputsFinais.length);
                
                $inputsFinais.each(function(index) {
                    var coluna = $(this).data('column');
                    var id = $(this).attr('id');
                    console.log('  Input manual', index + 1, '- ID:', id, 'Coluna:', coluna);
                });
            }, 200);
        }
    }
    
    // Adicionar evento de clique nos campos editáveis para edição direta
    $(document).on('click', '.editable-cell', function(e) {
        // Verificar se não está clicando em um input já aberto
        if ($(e.target).is('input') || $(e.target).closest('input').length > 0) {
            return;
        }
        
        // Verificar se a linha já está em modo de edição
        const linha = $(this).closest('tr');
        if (linha.hasClass('editing-row')) {
            return;
        }
        
        // Obter o ID do pagamento
        const idPagamento = linha.data('id');
        
        // Verificar se é uma linha válida (não é nova linha)
        if (idPagamento && !String(idPagamento).startsWith('novo_')) {
            FinanceiroClientesModal.editarInline(idPagamento);
        }
    });
    
    // Delegação de eventos para botões de salvar nova linha - COM VERIFICAÇÃO DE CONTEXTO
    $(document).on('click', '.btn-save', function(e) {
        // Verificar se estamos no modal financeiro
        const isInFinanceiroModal = $(this).closest('#modalFinanceiro').length > 0;
        
        if (!isInFinanceiroModal) {
            console.log('🔘 CLIQUE EM .btn-save FORA do modal financeiro - ignorando');
            return; // Não interceptar se não estiver no modal financeiro
        }
        
        e.preventDefault();
        e.stopPropagation();
        
        console.log('🔘 FINANCEIRO - CLIQUE SALVAR DETECTADO no modal financeiro');
        
        const linha = $(this).closest('tr');
        const novoId = linha.data('id');
        
        if (novoId && String(novoId).startsWith('novo_')) {
            console.log('🔘 CLIQUE SALVAR DETECTADO para ID:', novoId);
            FinanceiroClientesModal.salvarNovaLinha(novoId);
        }
        
        return false;
    });
    
    // Delegação de eventos para botões de cancelar nova linha - COM VERIFICAÇÃO DE CONTEXTO
    $(document).on('click', '.btn-cancel', function(e) {
        // Verificar se estamos no modal financeiro
        const isInFinanceiroModal = $(this).closest('#modalFinanceiro').length > 0;
        
        if (!isInFinanceiroModal) {
            console.log('🔘 CLIQUE EM .btn-cancel FORA do modal financeiro - ignorando');
            return; // Não interceptar se não estiver no modal financeiro
        }
        
        e.preventDefault();
        e.stopPropagation();
        
        const linha = $(this).closest('tr');
        const novoId = linha.data('id');
        
        if (novoId && String(novoId).startsWith('novo_')) {
            FinanceiroClientesModal.cancelarNovaLinha(novoId);
        }
        
        return false;
    });
    
    // Delegação de eventos para edição do nome do cliente
    $(document).on('click', '.cliente-nome-display', function(e) {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 CLIQUE DETECTADO no nome do cliente');
        FinanceiroClientesModal.editarNomeCliente();
        return false;
    });
    
    // Delegação de eventos para salvar nome do cliente
    $(document).on('click', '.cliente-salvar-btn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 CLIQUE SALVAR NOME CLIENTE DETECTADO');
        FinanceiroClientesModal.salvarNomeCliente();
        return false;
    });
    
    // Delegação de eventos para cancelar edição do nome do cliente
    $(document).on('click', '.cliente-cancelar-btn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 CLIQUE CANCELAR NOME CLIENTE DETECTADO');
        FinanceiroClientesModal.cancelarEdicaoNomeCliente();
        return false;
    });
    
    console.log('✅ Modal financeiro carregado com DataTable e filtros!');
    
    // Forçar exibição de todos os registros na primeira carga
    setTimeout(function() {
        if ($('#tabelaPagamentos').length > 0 && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
            const table = $('#tabelaPagamentos').DataTable();
            
            // Limpar todos os filtros
            table.search('').columns().search('').draw();
            
            // Garantir que a paginação está funcionando corretamente
            table.page.len(25).draw();
            
            console.log('🔄 Filtros limpos e registros recarregados');
        }
    }, 500);
    
    // Adicionar observer para proteger informações do pedido
    if (typeof MutationObserver !== 'undefined') {
        const protectorObserver = new MutationObserver(function(mutations) {
            let shouldProtect = false;
            mutations.forEach(function(mutation) {
                if (mutation.type === 'attributes' || mutation.type === 'childList') {
                    // Verificar se as informações do pedido foram afetadas
                    const target = mutation.target;
                    if (target && (target.id === 'info-pedido-container' || target.id === 'info-pedido-alert' || 
                        target.classList && (target.classList.contains('alert') || target.classList.contains('alert-info')))) {
                        shouldProtect = true;
                    }
                }
            });
            
            if (shouldProtect) {
                console.log('🔄 Mutação detectada nas informações do pedido, protegendo...');
                FinanceiroClientesModal.protegerInfoPedido();
            }
        });
        
        // Observar mudanças no container do modal
        const modalContainer = document.querySelector('.container-fluid');
        if (modalContainer) {
            protectorObserver.observe(modalContainer, {
                childList: true,
                subtree: true,
                attributes: true,
                attributeFilter: ['style', 'class', 'hidden']
            });
        }
    }
}

// Função para limpar estado do modal ao fechar
FinanceiroClientesModal.limparEstado = function() {
    console.log('🧹 Limpando estado do modal financeiro...');
    
    // Marcar como não carregado
    window.financeiroModalCarregado = false;
    
    // Destruir DataTable
    if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaPagamentos')) {
        try {
            $('#tabelaPagamentos').DataTable().destroy(false);
            console.log('✅ DataTable limpo ao fechar modal preservando dados');
        } catch (e) {
            console.log('⚠️ Erro ao limpar DataTable:', e);
        }
    }
    
    // Limpar wrappers
    $('.dataTables_wrapper').remove();
    $('#tabelaPagamentos').removeClass('dataTable');
    $('#tabelaPagamentos').removeAttr('role aria-describedby');
    
    // Limpar timeouts
    for (let i = 0; i < 10; i++) {
        clearTimeout(window['timeout_financeiro_col_' + i]);
    }
    
    // Limpar flags de processamento
    FinanceiroClientesModal.processandoSalvar = false;
};

// Detectar quando o modal é fechado (para qualquer modal Bootstrap)
$(document).on('hidden.bs.modal', function (e) {
    console.log('👋 Modal Bootstrap fechado, limpando estado financeiro...');
    if (window.financeiroModalCarregado && typeof FinanceiroClientesModal !== 'undefined') {
        FinanceiroClientesModal.limparEstado();
    }
});

// Detectar mudanças no DOM para reinicializar quando necessário
if (typeof MutationObserver !== 'undefined') {
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                // Verificar se o modal financeiro foi adicionado novamente
                const modalAdicionado = Array.from(mutation.addedNodes).some(node => 
                    node.nodeType === 1 && (
                        node.querySelector && node.querySelector('#tabelaPagamentos') ||
                        node.id === 'tabelaPagamentos'
                    )
                );
                
                if (modalAdicionado) {
                    console.log('🔄 Detectada re-adição do modal financeiro, reinicializando...');
                    setTimeout(function() {
                        if (!window.financeiroModalCarregado) {
                            $(document).trigger('ready');
                        }
                    }, 300);
                }
            }
        });
    });
    
    // Observar mudanças no documento
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
}