<?php
// Modal Genérico para CRUD - Sistema OutletByV
// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once 'dbconfig.php';
include_once 'includes/antisql.php';

// Verificar se a conexão foi estabelecida
if (!$conn) {
    echo json_encode(['success' => false, 'message' => 'Erro de conexão com banco de dados: ' . mysqli_connect_error()]);
    exit;
}

// Processar ações via POST - DEVE estar no início antes de qualquer saída HTML
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $acao = isset($_POST['acao']) ? $_POST['acao'] : '';
    $tabela = isset($_POST['tabela']) ? $_POST['tabela'] : '';
    $id_registro = isset($_POST['id_registro']) ? (int)$_POST['id_registro'] : 0;
    $campos = isset($_POST['campos']) ? $_POST['campos'] : [];
    
    // Validar tabela permitida (segurança)
    $tabelas_permitidas = ['vendas', 'financeiro_clientes', 'pedidos', 'usuarios', 'produtos'];
    if (!in_array($tabela, $tabelas_permitidas)) {
        echo json_encode(['success' => false, 'message' => 'Tabela não permitida.']);
        exit;
    }
    
    if ($acao === 'inserir') {
        if (!empty($campos) && !empty($tabela)) {
            // Construir query de inserção dinamicamente
            $colunas = array_keys($campos);
            $placeholders = str_repeat('?,', count($colunas) - 1) . '?';
            $sql = "INSERT INTO {$tabela} (" . implode(',', $colunas) . ") VALUES ({$placeholders})";
            
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $tipos = '';
                $valores = [];
                
                foreach ($campos as $valor) {
                    if (is_numeric($valor)) {
                        $tipos .= 'd';
                        $valores[] = (float)str_replace(',', '.', $valor);
                    } else {
                        $tipos .= 's';
                        $valores[] = $valor;
                    }
                }
                
                $stmt->bind_param($tipos, ...$valores);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Registro adicionado com sucesso!', 'id_registro' => $conn->insert_id]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao adicionar registro: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Campos obrigatórios não preenchidos.']);
            exit;
        }
    } elseif ($acao === 'alterar' && $id_registro > 0) {
        if (!empty($campos) && !empty($tabela)) {
            // Construir query de atualização dinamicamente
            $sets = [];
            foreach (array_keys($campos) as $coluna) {
                $sets[] = "{$coluna} = ?";
            }
            $sql = "UPDATE {$tabela} SET " . implode(', ', $sets) . " WHERE id = ?";
            
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $tipos = '';
                $valores = [];
                
                foreach ($campos as $valor) {
                    if (is_numeric($valor)) {
                        $tipos .= 'd';
                        $valores[] = (float)str_replace(',', '.', $valor);
                    } else {
                        $tipos .= 's';
                        $valores[] = $valor;
                    }
                }
                
                $tipos .= 'i';
                $valores[] = $id_registro;
                
                $stmt->bind_param($tipos, ...$valores);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Registro atualizado com sucesso!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao atualizar registro: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Campos obrigatórios não preenchidos.']);
            exit;
        }
    } elseif ($acao === 'excluir' && $id_registro > 0) {
        $sql = "DELETE FROM {$tabela} WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_registro);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Registro excluído com sucesso!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao excluir registro: ' . $stmt->error]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    }
}

// Processar ação buscar_registro via GET
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['acao']) && $_GET['acao'] === 'buscar_registro') {
    header('Content-Type: application/json');
    
    $id_registro = isset($_GET['id_registro']) ? (int)$_GET['id_registro'] : 0;
    $tabela = isset($_GET['tabela']) ? $_GET['tabela'] : '';
    
    // Validar tabela permitida (segurança)
    $tabelas_permitidas = ['vendas', 'financeiro_clientes', 'pedidos', 'usuarios', 'produtos'];
    if (!in_array($tabela, $tabelas_permitidas)) {
        echo json_encode(['success' => false, 'message' => 'Tabela não permitida.']);
        exit;
    }
    
    if ($id_registro > 0 && !empty($tabela)) {
        $sql = "SELECT * FROM {$tabela} WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_registro);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $registro = $result->fetch_assoc();
                
                if ($registro) {
                    echo json_encode(['success' => true, 'data' => $registro]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Registro não encontrado.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao buscar registro.']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Parâmetros inválidos.']);
    }
    exit;
}

// Receber parâmetros para exibição
$tabela = isset($_GET['tabela']) ? $_GET['tabela'] : 'vendas';
$id_pai = isset($_GET['id_pai']) ? (int)$_GET['id_pai'] : 0;
$campo_pai = isset($_GET['campo_pai']) ? $_GET['campo_pai'] : 'id_pedido';
$titulo = isset($_GET['titulo']) ? $_GET['titulo'] : 'Registros';
$campos_config = isset($_GET['campos']) ? json_decode($_GET['campos'], true) : [];

// Configurações padrão por tabela
$configuracoes_tabela = [
    'vendas' => [
        'titulo' => 'Vendas do Pedido',
        'campos' => [
            'produto' => ['tipo' => 'text', 'label' => 'Produto', 'obrigatorio' => true],
            'valor' => ['tipo' => 'money', 'label' => 'Valor', 'obrigatorio' => true],
            'data2' => ['tipo' => 'date', 'label' => 'Data', 'obrigatorio' => true]
        ]
    ],
    'financeiro_clientes' => [
        'titulo' => 'Financeiro do Cliente',
        'campos' => [
            'valor' => ['tipo' => 'money', 'label' => 'Valor', 'obrigatorio' => true],
            'data2' => ['tipo' => 'date', 'label' => 'Data Pagamento', 'obrigatorio' => true]
        ]
    ]
];

// Usar configuração específica ou padrão
$config = isset($configuracoes_tabela[$tabela]) ? $configuracoes_tabela[$tabela] : [
    'titulo' => $titulo,
    'campos' => $campos_config
];

// Buscar registros da tabela
$registros = [];
if ($id_pai > 0) {
    $sql = "SELECT * FROM {$tabela} WHERE {$campo_pai} = ? ORDER BY id DESC";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("i", $id_pai);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $registros[] = $row;
            }
        }
    }
}

// Função para formatar data
function formatarData($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return 'Sem data';
    }
    
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        return $data;
    }
    
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        $timestamp = strtotime($data);
        if ($timestamp && $timestamp > 0) {
            return date('d/m/Y', $timestamp);
        }
    }
    
    return $data;
}

// Função para converter data para input date
function converterDataParaInput($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return '';
    }
    
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        return $data;
    }
    
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        $partes = explode('/', $data);
        return $partes[2] . '-' . $partes[1] . '-' . $partes[0];
    }
    
    $timestamp = strtotime($data);
    if ($timestamp && $timestamp > 0) {
        return date('Y-m-d', $timestamp);
    }
    
    return '';
}
?>

<!-- Informações do Registro Pai -->
<?php if ($id_pai > 0): ?>
<div class="row mb-3">
    <div class="col-12">
        <div class="alert alert-info">
            <strong><?php echo ucfirst($campo_pai); ?>:</strong> <?php echo htmlspecialchars($id_pai); ?>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Mensagens de alerta -->
<div class="row mb-4">
    <div class="col-12">
        <div id="alertMessages"></div>
    </div>
</div>

<!-- Lista de Registros -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0"><?php echo htmlspecialchars($config['titulo']); ?></h5>
                <div class="d-flex align-items-center gap-2">
                    <button type="button" class="btn btn-success btn-sm" onclick="adicionarNovaLinha()" title="Adicionar Novo Registro">
                        <i class="fas fa-plus"></i> Novo Registro
                    </button>
                </div>
            </div>
            <div class="card-body">
                <?php if (empty($registros)): ?>
                <div class="text-center text-muted py-4">
                    <i class="fas fa-database fa-3x mb-3"></i>
                    <p>Nenhum registro encontrado.</p>
                </div>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover" id="tabelaRegistros">
                        <thead class="table-dark">
                            <tr>
                                <th>ID</th>
                                <?php foreach ($config['campos'] as $campo => $info): ?>
                                <th><?php echo htmlspecialchars($info['label']); ?></th>
                                <?php endforeach; ?>
                                <th width="120">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($registros as $registro): ?>
                            <tr data-id="<?php echo $registro['id']; ?>">
                                <td><?php echo $registro['id']; ?></td>
                                <?php foreach ($config['campos'] as $campo => $info): ?>
                                <td class="editable-cell" data-field="<?php echo $campo; ?>" data-original="<?php echo htmlspecialchars($registro[$campo] ?? ''); ?>">
                                    <span class="cell-display">
                                        <?php 
                                        $valor = $registro[$campo] ?? '';
                                        if ($info['tipo'] === 'money') {
                                            echo 'R$ ' . number_format($valor, 2, ',', '.');
                                        } elseif ($info['tipo'] === 'date') {
                                            echo formatarData($valor);
                                        } else {
                                            echo htmlspecialchars($valor);
                                        }
                                        ?>
                                    </span>
                                    <?php if ($info['tipo'] === 'money'): ?>
                                    <input type="text" class="cell-input form-control form-control-sm valor-input" value="<?php echo $registro[$campo] ?? ''; ?>" style="display: none;">
                                    <?php elseif ($info['tipo'] === 'date'): ?>
                                    <input type="date" class="cell-input form-control form-control-sm" value="<?php echo converterDataParaInput($registro[$campo] ?? ''); ?>" style="display: none;">
                                    <?php else: ?>
                                    <input type="text" class="cell-input form-control form-control-sm" value="<?php echo htmlspecialchars($registro[$campo] ?? ''); ?>" style="display: none;">
                                    <?php endif; ?>
                                </td>
                                <?php endforeach; ?>
                                <td>
                                    <button type="button" class="btn btn-sm btn-success btn-save" onclick="salvarEdicaoInline(<?php echo $registro['id']; ?>)" title="Salvar" style="display: none;">
                                        <i class="fas fa-check"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="cancelarEdicaoInline(<?php echo $registro['id']; ?>)" title="Cancelar" style="display: none;">
                                        <i class="fas fa-times"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-warning btn-edit" onclick="editarInline(<?php echo $registro['id']; ?>)" title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" onclick="excluirRegistro(<?php echo $registro['id']; ?>)" title="Excluir">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- CSS Específico do Modal -->
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
<link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap4.min.css">

<style>
/* Estilos para edição inline */
.editable-cell { cursor: pointer; position: relative; }
.editable-cell:hover { background-color: #f8f9fa; }
.editable-cell.editing { background-color: #fff3cd; cursor: default; }
.cell-input { width: 100%; border: 2px solid #007bff; background-color: #fff; }
.cell-input:focus { border-color: #0056b3; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
.valor-input { text-align: right; }
.editing-row { background-color: #fff3cd !important; }
.nova-linha { background-color: #d1ecf1 !important; }

/* Estilos para os filtros de coluna */
.footer-search-input { width: 100%; padding: 4px 8px; font-size: 13px; border: 1px solid #ced4da; border-radius: 4px; }
.footer-search-input:focus { border-color: #80bdff; outline: 0; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
#tabelaRegistros tfoot { background-color: #f8f9fa; }
#tabelaRegistros tfoot th { padding: 8px 4px; font-weight: normal; }
</style>

<!-- JavaScript -->
<script type="text/javascript" src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script type="text/javascript" src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
// Variáveis globais
const TABELA = '<?php echo $tabela; ?>';
const ID_PAI = <?php echo $id_pai; ?>;
const CAMPO_PAI = '<?php echo $campo_pai; ?>';
const CAMPOS_CONFIG = <?php echo json_encode($config['campos']); ?>;

// Função para editar linha inline
function editarInline(idRegistro) {
    cancelarTodasEdicoes();
    
    const linha = $(`tr[data-id="${idRegistro}"]`);
    linha.addClass('editing-row');
    
    // Preparar valores antes de mostrar os inputs
    linha.find('.editable-cell').each(function() {
        const campo = $(this).data('field');
        const valorOriginal = $(this).data('original');
        const input = $(this).find('.cell-input');
        const configCampo = CAMPOS_CONFIG[campo];
        
        if (configCampo && configCampo.tipo === 'money' && valorOriginal) {
            const valorFormatado = parseFloat(valorOriginal).toLocaleString('pt-BR', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
            input.val(valorFormatado);
        }
    });
    
    linha.find('.cell-display').hide();
    linha.find('.cell-input').show();
    linha.find('.btn-edit, .btn-danger').hide();
    linha.find('.btn-save, .btn-cancel').show();
    
    // Aplicar máscaras nos campos de valor
    linha.find('.valor-input').each(function() {
        aplicarMascaraValorInline($(this));
    });
    
    linha.find('input').first().focus();
}

// Função para cancelar edição
function cancelarEdicaoInline(idRegistro) {
    const linha = $(`tr[data-id="${idRegistro}"]`);
    linha.removeClass('editing-row');
    
    // Restaurar valores originais
    linha.find('.editable-cell').each(function() {
        const valorOriginal = $(this).data('original');
        $(this).find('.cell-input').val(valorOriginal);
    });
    
    linha.find('.cell-input').hide();
    linha.find('.cell-display').show();
    linha.find('.btn-save, .btn-cancel').hide();
    linha.find('.btn-edit, .btn-danger').show();
}

// Função para salvar edição
function salvarEdicaoInline(idRegistro) {
    const linha = $(`tr[data-id="${idRegistro}"]`);
    const campos = {};
    let camposValidos = true;
    
    linha.find('.editable-cell').each(function() {
        const campo = $(this).data('field');
        const valor = $(this).find('.cell-input').val();
        const configCampo = CAMPOS_CONFIG[campo];
        
        if (configCampo && configCampo.obrigatorio && !valor.trim()) {
            camposValidos = false;
            return false;
        }
        
        if (configCampo && configCampo.tipo === 'money') {
            campos[campo] = valor.replace(/\./g, '').replace(',', '.');
        } else {
            campos[campo] = valor;
        }
    });
    
    if (!camposValidos) {
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Todos os campos obrigatórios devem ser preenchidos.' });
        return;
    }
    
    // Adicionar campo pai
    campos[CAMPO_PAI] = ID_PAI;
    
    fetch('modal_generico.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'alterar',
            tabela: TABELA,
            id_registro: idRegistro,
            campos: JSON.stringify(campos)
        })
    })
    .then(response => response.json())
    .then(responseData => {
        if (responseData.success) {
            // Atualizar displays
            linha.find('.editable-cell').each(function() {
                const campo = $(this).data('field');
                const valor = campos[campo];
                const configCampo = CAMPOS_CONFIG[campo];
                const display = $(this).find('.cell-display');
                
                if (configCampo && configCampo.tipo === 'money') {
                    display.text('R$ ' + parseFloat(valor).toLocaleString('pt-BR', {
                        minimumFractionDigits: 2, maximumFractionDigits: 2
                    }));
                } else if (configCampo && configCampo.tipo === 'date') {
                    display.text(formatarDataParaExibicao(valor));
                } else {
                    display.text(valor);
                }
                
                // Atualizar dados originais
                $(this).data('original', valor);
            });
            
            // Sair do modo de edição
            linha.removeClass('editing-row');
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            linha.find('.btn-save, .btn-cancel').hide();
            linha.find('.btn-edit, .btn-danger').show();
            
            Swal.fire({ icon: 'success', title: 'Sucesso!', text: responseData.message, timer: 1500, showConfirmButton: false });
        } else {
            Swal.fire({ icon: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para formatar data para exibição
function formatarDataParaExibicao(data) {
    if (!data || data === '0000-00-00' || data === '1970-01-01') {
        return 'Sem data';
    }
    
    if (/^\d{2}\/\d{2}\/\d{4}$/.test(data)) {
        return data;
    }
    
    if (/^\d{4}-\d{2}-\d{2}$/.test(data)) {
        try {
            const [ano, mes, dia] = data.split('-');
            return `${dia}/${mes}/${ano}`;
        } catch (e) {
            return data;
        }
    }
    
    try {
        const dateObj = new Date(data + 'T00:00:00');
        if (!isNaN(dateObj.getTime())) {
            return dateObj.toLocaleDateString('pt-BR');
        }
    } catch (e) {
        // Se falhar, retorna a data original
    }
    
    return data;
}

// Função para cancelar todas as edições
function cancelarTodasEdicoes() {
    $('.editing-row').each(function() {
        const idRegistro = $(this).data('id');
        if (typeof idRegistro === 'string' && idRegistro.startsWith('novo_')) {
            $(this).remove();
        } else {
            cancelarEdicaoInline(idRegistro);
        }
    });
}

// Função para aplicar máscara de valor
function aplicarMascaraValorInline(campo) {
    campo.off('.mascaraInline');
    
    campo.on('input.mascaraInline', function() {
        let valor = this.value.replace(/\D/g, '');
        if (valor.length === 0) { this.value = ''; return; }
        
        if (valor.length === 1) {
            this.value = '0,0' + valor;
            return;
        }
        
        if (valor.length === 2) {
            this.value = '0,' + valor;
            return;
        }
        
        const parteDecimal = valor.slice(-2);
        let parteInteira = valor.slice(0, -2);
        
        if (parteInteira.length > 3) {
            parteInteira = parteInteira.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        }
        
        this.value = parteInteira + ',' + parteDecimal;
    });
    
    campo.on('focus.mascaraInline', function() {
        setTimeout(() => this.select(), 10);
    });
}

// Função para adicionar nova linha
function adicionarNovaLinha() {
    cancelarTodasEdicoes();
    
    const novoId = 'novo_' + Date.now();
    let colunasCampos = '';
    
    Object.keys(CAMPOS_CONFIG).forEach(campo => {
        const config = CAMPOS_CONFIG[campo];
        let inputHtml = '';
        
        if (config.tipo === 'money') {
            inputHtml = '<input type="text" class="form-control form-control-sm cell-input valor-input" placeholder="0,00" style="display: block;">';
        } else if (config.tipo === 'date') {
            inputHtml = '<input type="date" class="form-control form-control-sm cell-input" style="display: block;">';
        } else {
            inputHtml = `<input type="text" class="form-control form-control-sm cell-input" placeholder="Digite ${config.label.toLowerCase()}" style="display: block;">`;
        }
        
        colunasCampos += `
            <td class="editable-cell" data-field="${campo}" data-original="">
                <span class="cell-display" style="display: none;"></span>
                ${inputHtml}
            </td>`;
    });
    
    const novaLinha = `
        <tr data-id="${novoId}" class="editing-row nova-linha">
            <td>Novo</td>
            ${colunasCampos}
            <td>
                <button type="button" class="btn btn-sm btn-success btn-save" onclick="salvarNovoRegistro('${novoId}')" title="Salvar" style="display: block;">
                    <i class="fas fa-check"></i>
                </button>
                <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="cancelarNovoRegistro('${novoId}')" title="Cancelar" style="display: block;">
                    <i class="fas fa-times"></i>
                </button>
            </td>
        </tr>`;
    
    $('#tabelaRegistros tbody').prepend(novaLinha);
    
    // Aplicar máscaras nos campos de valor
    $(`tr[data-id="${novoId}"] .valor-input`).each(function() {
        aplicarMascaraValorInline($(this));
    });
    
    $(`tr[data-id="${novoId}"] input`).first().focus();
}

// Função para salvar novo registro
function salvarNovoRegistro(novoId) {
    const linha = $(`tr[data-id="${novoId}"]`);
    const campos = {};
    let camposValidos = true;
    
    linha.find('.editable-cell').each(function() {
        const campo = $(this).data('field');
        const valor = $(this).find('.cell-input').val();
        const configCampo = CAMPOS_CONFIG[campo];
        
        if (configCampo && configCampo.obrigatorio && !valor.trim()) {
            camposValidos = false;
            return false;
        }
        
        if (configCampo && configCampo.tipo === 'money') {
            campos[campo] = valor.replace(/\./g, '').replace(',', '.');
        } else {
            campos[campo] = valor;
        }
    });
    
    if (!camposValidos) {
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Todos os campos obrigatórios devem ser preenchidos.' });
        return;
    }
    
    // Adicionar campo pai
    campos[CAMPO_PAI] = ID_PAI;
    
    fetch('modal_generico.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'inserir',
            tabela: TABELA,
            campos: JSON.stringify(campos)
        })
    })
    .then(response => response.json())
    .then(responseData => {
        if (responseData.success) {
            // Recarregar a página para mostrar o novo registro
            location.reload();
        } else {
            Swal.fire({ icon: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para cancelar novo registro
function cancelarNovoRegistro(novoId) {
    $(`tr[data-id="${novoId}"]`).remove();
}

// Função para excluir registro
function excluirRegistro(idRegistro) {
    Swal.fire({
        title: 'Tem certeza?',
        text: 'Esta ação não pode ser desfeita!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sim, excluir!',
        cancelButtonText: 'Cancelar'
    }).then((result) => {
        if (result.isConfirmed) {
            fetch('modal_generico.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    acao: 'excluir',
                    tabela: TABELA,
                    id_registro: idRegistro
                })
            })
            .then(response => response.json())
            .then(responseData => {
                if (responseData.success) {
                    $(`tr[data-id="${idRegistro}"]`).remove();
                    Swal.fire({ icon: 'success', title: 'Excluído!', text: responseData.message, timer: 1500, showConfirmButton: false });
                } else {
                    Swal.fire({ icon: 'error', title: 'Erro!', text: responseData.message });
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                Swal.fire({ icon: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
            });
        }
    });
}

// Evento de clique para edição direta
$(document).on('click', '.editable-cell', function() {
    const linha = $(this).closest('tr');
    const idRegistro = linha.data('id');
    
    // Verificar se já não está em edição e se não é uma linha nova
    if (!linha.hasClass('editing-row') && 
        typeof idRegistro === 'number' && 
        !$(this).find('.cell-input').is(':visible')) {
        editarInline(idRegistro);
    }
});

// Inicializar DataTable
$(document).ready(function() {
    if ($('#tabelaRegistros').length) {
        $('#tabelaRegistros').DataTable({
            "language": {
                "url": "datatable/pt-BR.json"
            },
            "responsive": true,
            "processing": false,
            "pageLength": 25,
            "order": [[0, "desc"]]
        });
    }
});
</script>