<?php
// TESTE DEFINITIVO - VERIFICAR SE ESTE ARQUIVO ESTÁ SENDO EXECUTADO
error_log("🚨 VENDAS_MODAL.PHP - ARQUIVO INICIADO - TIMESTAMP: " . date('Y-m-d H:i:s.u'));
error_log("🚨 ARQUIVO ATUAL: " . __FILE__);
error_log("🚨 SCRIPT NAME: " . $_SERVER['SCRIPT_NAME']);

// Habilitar exibição de erros para debug
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once 'dbconfig.php';
include_once 'includes/antisql.php';

// Verificar se a conexão foi estabelecida
if (!$conn) {
    echo json_encode(['success' => false, 'message' => 'Erro de conexão com banco de dados: ' . mysqli_connect_error()]);
    exit;
}

// Processar ações via POST - DEVE estar no início antes de qualquer saída HTML
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // LOG DE DEBUG - INÍCIO
    error_log("=== VENDAS_MODAL.PHP - REQUISIÇÃO POST RECEBIDA ===");
    error_log("POST DATA: " . print_r($_POST, true));
    error_log("TIMESTAMP: " . date('Y-m-d H:i:s'));
    error_log("USER_AGENT: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'N/A'));
    error_log("REFERER: " . ($_SERVER['HTTP_REFERER'] ?? 'N/A'));
    // LOG DE DEBUG - FIM
    
    header('Content-Type: application/json');
    
    $acao = isset($_POST['acao']) ? $_POST['acao'] : '';
    $id_pedido = isset($_POST['id_pedido']) ? (int)$_POST['id_pedido'] : 0;
    $id_cliente = isset($_POST['id_cliente']) ? (int)$_POST['id_cliente'] : 0;
    $id_vendas = isset($_POST['id_vendas']) ? (int)$_POST['id_vendas'] : 0;
    
    // LOG ESPECÍFICO PARA EXCLUSÃO
    if ($acao === 'excluir') {
        error_log("🗑️ VENDAS_MODAL.PHP - EXCLUSÃO DETECTADA!");
        error_log("🗑️ id_vendas: $id_vendas");
        error_log("🗑️ id_pedido: $id_pedido");
        error_log("🗑️ id_cliente: $id_cliente");
        error_log("🗑️ Condição (id_vendas > 0): " . ($id_vendas > 0 ? 'TRUE' : 'FALSE'));
    }
    
    if ($acao === 'inserir') {
        error_log("🚨 VENDAS_MODAL.PHP - SEÇÃO INSERIR EXECUTADA - TIMESTAMP: " . date('Y-m-d H:i:s.u'));
        
        $produto = isset($_POST['produto']) ? $_POST['produto'] : '';
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : 0;
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        error_log("🚨 VENDAS_MODAL.PHP - DADOS EXTRAÍDOS: produto=$produto, valor=$valor, data=$data");
        
        // Obter hora atual (data2 recebe a data digitada pelo usuário)
        $hora_atual = date('H:i:s');
        
        if (!empty($produto) && $valor > 0) {
            error_log("🚨 VENDAS_MODAL.PHP - VALIDAÇÃO OK, EXECUTANDO INSERT NA TABELA VENDAS");
            
            $sql = "INSERT INTO vendas (id_pedido, produto, valor, data2, data, hora) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("isdsss", $id_pedido, $produto, $valor, $data, $data, $hora_atual);
                
                if ($stmt->execute()) {
                    $insert_id = $conn->insert_id;
                    error_log("🚨 VENDAS_MODAL.PHP - INSERT EXECUTADO COM SUCESSO! ID: $insert_id");
                    echo json_encode(['success' => true, 'message' => 'Venda adicionada com sucesso!', 'id_vendas' => $insert_id]);
                } else {
                    error_log("🚨 VENDAS_MODAL.PHP - ERRO NO EXECUTE: " . $stmt->error);
                    echo json_encode(['success' => false, 'message' => 'Erro ao adicionar venda: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Produto e valor são obrigatórios.']);
            exit;
        }
        } elseif ($acao === 'alterar' && $id_vendas > 0) {
        $produto = isset($_POST['produto']) ? $_POST['produto'] : '';
        $valor = isset($_POST['valor']) ? str_replace(',', '.', $_POST['valor']) : 0;
        $data = isset($_POST['data']) ? $_POST['data'] : date('Y-m-d');
        
        // Obter hora atual (data2 recebe a data digitada pelo usuário)
        $hora_atual = date('H:i:s');
        
        if (!empty($produto) && $valor > 0) {
            $sql = "UPDATE vendas SET produto = ?, valor = ?, data2 = ?, data = ?, hora = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            
            if ($stmt) {
                $stmt->bind_param("sdsssi", $produto, $valor, $data, $data, $hora_atual, $id_vendas);
                
                if ($stmt->execute()) {
                    echo json_encode(['success' => true, 'message' => 'Venda atualizada com sucesso!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Erro ao atualizar venda: ' . $stmt->error]);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
            }
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Produto e valor são obrigatórios.']);
            exit;
        }
    } elseif ($acao === 'excluir' && $id_vendas > 0) {
        error_log("🗑️ VENDAS_MODAL.PHP - SEÇÃO EXCLUIR EXECUTADA - ID: $id_vendas - TIMESTAMP: " . date('Y-m-d H:i:s.u'));
        
        $sql = "DELETE FROM vendas WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_vendas);
            
            if ($stmt->execute()) {
                $rows_affected = $stmt->affected_rows;
                error_log("🗑️ VENDAS_MODAL.PHP - DELETE EXECUTADO COM SUCESSO! Linhas afetadas: $rows_affected");
                echo json_encode(['success' => true, 'message' => 'Venda excluída com sucesso!']);
            } else {
                error_log("🗑️ VENDAS_MODAL.PHP - ERRO NO EXECUTE: " . $stmt->error);
                echo json_encode(['success' => false, 'message' => 'Erro ao excluir venda: ' . $stmt->error]);
            }
        } else {
            error_log("🗑️ VENDAS_MODAL.PHP - ERRO NO PREPARE: " . $conn->error);
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta: ' . $conn->error]);
        }
        exit;
    } else {
        // Log para ações não processadas
        error_log("❌ VENDAS_MODAL.PHP - AÇÃO NÃO PROCESSADA!");
        error_log("❌ Ação recebida: '$acao'");
        error_log("❌ id_vendas: $id_vendas");
        error_log("❌ Condições: acao='" . $acao . "', id_vendas > 0 = " . ($id_vendas > 0 ? 'TRUE' : 'FALSE'));
        echo json_encode(['success' => false, 'message' => 'Ação não reconhecida ou parâmetros inválidos.']);
        exit;
    }
}

// Processar ação buscar_venda via GET
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['acao']) && $_GET['acao'] === 'buscar_venda') {
    header('Content-Type: application/json');
    
    $id_vendas = isset($_GET['id_vendas']) ? (int)$_GET['id_vendas'] : 0;
    
    if ($id_vendas > 0) {
        $sql = "SELECT * FROM vendas WHERE id = ?";
        $stmt = $conn->prepare($sql);
        
        if ($stmt) {
            $stmt->bind_param("i", $id_vendas);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                $venda = $result->fetch_assoc();
                
                if ($venda) {
                    $valor_formatado = number_format($venda['valor'], 2, ',', '.');
                    
                    echo json_encode([
                        'success' => true,
                        'data' => [
                            'produto' => $venda['produto'],
                            'valor_formatado' => $valor_formatado,
                            'data_venda' => $venda['data2']
                        ]
                    ]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Venda não encontrada.']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Erro ao buscar venda.']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Erro ao preparar consulta.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'ID da venda inválido.']);
    }
    exit;
}

// Receber parâmetros
$id_pedido = isset($_GET['id_pedido']) ? (int)$_GET['id_pedido'] : 0;
$id_cliente = isset($_GET['id_cliente']) ? (int)$_GET['id_cliente'] : 0;

// Buscar informações do pedido e cliente
$info_pedido = null;
if ($id_pedido > 0 && $id_cliente > 0) {
    $sql = "SELECT p.id as pedido_id, p.data, u.nome as cliente_nome 
            FROM pedidos p 
            LEFT JOIN usuarios u ON p.id_cliente = u.id 
            WHERE p.id = ? AND p.id_cliente = ?";
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("ii", $id_pedido, $id_cliente);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $info_pedido = $result->fetch_assoc();
        }
    }
}

// Buscar vendas do pedido
$vendas = [];
if ($id_pedido > 0) {
    $sql = "SELECT * FROM vendas WHERE id_pedido = ? ORDER BY id DESC, data2 DESC";
    error_log("🔍 VENDAS_MODAL.PHP - SQL EXECUTADO: $sql com id_pedido=$id_pedido");
    $stmt = $conn->prepare($sql);
    
    if ($stmt) {
        $stmt->bind_param("i", $id_pedido);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $vendas[] = $row;
            }
            
            // LOG DE DEBUG - Quantos registros foram encontrados
            error_log("🔍 VENDAS_MODAL.PHP - VENDAS ENCONTRADAS: " . count($vendas) . " para id_pedido=$id_pedido");
            
            // LOG DE DEBUG - Ordem dos IDs para verificar a ordenação
            $ids_ordem = array_map(function($venda) { return $venda['id']; }, $vendas);
            error_log("📋 VENDAS_MODAL.PHP - ORDEM DOS IDs: " . implode(', ', $ids_ordem));
            error_log("📅 VENDAS_MODAL.PHP - ORDEM DAS DATAS: " . implode(', ', array_map(function($venda) { return $venda['data2']; }, $vendas)));
        } else {
            error_log("❌ VENDAS_MODAL.PHP - ERRO AO EXECUTAR CONSULTA: " . $stmt->error);
        }
    } else {
        error_log("❌ VENDAS_MODAL.PHP - ERRO AO PREPARAR CONSULTA: " . $conn->error);
    }
}

// Calcular total de vendas
$total_vendas = 0;
foreach ($vendas as $venda) {
    $total_vendas += $venda['valor'];
}

// Função para formatar data
function formatarData($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return 'Sem data';
    }
    
    // Se já está no formato dd/mm/yyyy
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        return $data;
    }
    
    // Se está no formato yyyy-mm-dd
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        $timestamp = strtotime($data);
        if ($timestamp && $timestamp > 0) {
            return date('d/m/Y', $timestamp);
        }
    }
    
    return $data;
}

// Função para converter data para input date
function converterDataParaInput($data) {
    if (empty($data) || $data === '0000-00-00' || $data === '1970-01-01') {
        return '';
    }
    
    // Se já está no formato yyyy-mm-dd
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data)) {
        return $data;
    }
    
    // Se está no formato dd/mm/yyyy
    if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $data)) {
        $partes = explode('/', $data);
        return $partes[2] . '-' . $partes[1] . '-' . $partes[0];
    }
    
    // Tentar converter outros formatos
    $timestamp = strtotime($data);
    if ($timestamp && $timestamp > 0) {
        return date('Y-m-d', $timestamp);
    }
    
    return '';
}
?>

<!-- Informações do Pedido -->
<?php if ($info_pedido): ?>
<div class="row mb-3">
    <div class="col-12">
        <div class="alert alert-info">
            <strong>Pedido ID:</strong> <?php echo htmlspecialchars($info_pedido['pedido_id']); ?> | 
            <strong>Data:</strong> <?php echo formatarData($info_pedido['data']); ?> |
            <strong>Cliente:</strong> <?php echo htmlspecialchars($info_pedido['cliente_nome']); ?>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Mensagens de alerta -->
<div class="row mb-4">
    <div class="col-12">
        <div id="alertMessages"></div>
    </div>
</div>

<!-- Lista de Vendas -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Vendas do Pedido</h5>
                <div class="d-flex align-items-center gap-2">
                    <button type="button" class="btn btn-success btn-sm" onclick="VendasModal.adicionarNovaLinha()" title="Adicionar Nova Venda">
                        <i class="fas fa-plus"></i> Nova Venda
                    </button>
                    <span class="badge bg-primary fs-6">Total: R$ <?php echo number_format($total_vendas, 2, ',', '.'); ?></span>
                </div>
            </div>
            <div class="card-body">
                <?php if (empty($vendas)): ?>
                <div class="text-center text-muted py-4 mb-3">
                    <i class="fas fa-shopping-cart fa-3x mb-3"></i>
                    <p>Nenhuma venda encontrada para este pedido.</p>
                    <p class="text-info">Clique no botão "Nova Venda" acima para adicionar a primeira venda.</p>
                </div>
                
                <!-- Tabela vazia para permitir adição de vendas -->
                <div class="table-responsive" style="display: none;">
                    <table class="table table-striped table-hover" id="tabelaVendas">
                        <thead class="table-dark">
                            <tr>
                                <th>Nº</th>
                                <th>Produto</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th width="120">Ações</th>
                            </tr>
                        </thead>
                        <tfoot>
                            <tr>
                                <th>Nº</th>
                                <th>Produto</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th>Ações</th>
                            </tr>
                        </tfoot>
                        <tbody>
                            <!-- Tbody vazio - vendas serão adicionadas dinamicamente -->
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover" id="tabelaVendas">
                        <thead class="table-dark">
                            <tr>
                                <th>Nº</th>
                                <th>Produto</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th width="120">Ações</th>
                            </tr>
                        </thead>
                        <tfoot>
                            <tr>
                                <th>Nº</th>
                                <th>Produto</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th>Ações</th>
                            </tr>
                        </tfoot>
                        <tbody>
                            <?php foreach ($vendas as $venda): ?>
                            <tr data-id="<?php echo $venda['id']; ?>">
                                <td><?php echo $venda['id']; ?></td>
                                <td class="editable-cell" data-field="produto" data-original="<?php echo htmlspecialchars($venda['produto']); ?>">
                                    <span class="cell-display"><?php echo htmlspecialchars($venda['produto']); ?></span>
                                    <input type="text" class="cell-input form-control form-control-sm" value="<?php echo htmlspecialchars($venda['produto']); ?>" style="display: none;">
                                </td>
                                <td class="editable-cell" data-field="valor" data-original="<?php echo $venda['valor']; ?>">
                                    <span class="cell-display">R$ <?php echo number_format($venda['valor'], 2, ',', '.'); ?></span>
                                    <input type="text" class="cell-input form-control form-control-sm valor-input" value="<?php echo $venda['valor']; ?>" style="display: none;">
                                </td>
                                <td class="editable-cell" data-field="data2" data-original="<?php echo $venda['data2']; ?>">
                                    <span class="cell-display"><?php echo formatarData($venda['data2']); ?></span>
                                    <input type="date" class="cell-input form-control form-control-sm" value="<?php echo converterDataParaInput($venda['data2']); ?>" style="display: none;">
                                </td>
                                <td>
                                    <button type="button" class="btn btn-sm btn-success btn-save" onclick="VendasModal.salvarEdicaoInline(<?php echo $venda['id']; ?>)" title="Salvar" style="display: none;">
                                        <i class="fas fa-check"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-secondary btn-cancel" onclick="VendasModal.cancelarEdicaoInline(<?php echo $venda['id']; ?>)" title="Cancelar" style="display: none;">
                                        <i class="fas fa-times"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-warning btn-edit" onclick="VendasModal.editarInline(<?php echo $venda['id']; ?>)" title="Editar">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" onclick="VendasModal.excluirVenda(<?php echo $venda['id']; ?>)" title="Excluir">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- CSS Específico do Modal -->

<style>
/* Estilos mínimos específicos para edição inline */
.editable-cell { cursor: pointer; position: relative; }
.editable-cell:hover { background-color: #f8f9fa; }
.editable-cell.editing { background-color: #fff3cd; cursor: default; }
.cell-input { width: 100%; border: 2px solid #007bff; background-color: #fff; }
.cell-input:focus { border-color: #0056b3; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
.valor-input { text-align: right; }
.editing-row { background-color: #fff3cd !important; }
.nova-linha { background-color: #d1ecf1 !important; }

/* Estilos para os filtros de coluna */
.md-input-wrapper { margin-bottom: 0; }
.footer-search-input-vendas { width: 100%; padding: 4px 8px; font-size: 13px; border: 1px solid #ced4da; border-radius: 4px; }
.footer-search-input-vendas:focus { border-color: #80bdff; outline: 0; box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25); }
#tabelaVendas tfoot { background-color: #f8f9fa; }
#tabelaVendas tfoot th { padding: 8px 4px; font-weight: normal; }

/* ========================================
   CORREÇÃO Z-INDEX SWEETALERT2 - VENDAS MODAL
   ======================================== */

/* Container principal do SweetAlert2 - deve ser o mais alto de todos */
.swal2-container {
    z-index: 99999 !important;
    position: fixed !important;
    top: 0 !important;
    left: 0 !important;
    right: 0 !important;
    bottom: 0 !important;
}

/* Popup do SweetAlert2 */
.swal2-popup {
    z-index: 100000 !important;
    position: relative !important;
}

/* Backdrop do SweetAlert2 */
.swal2-backdrop {
    z-index: 99998 !important;
    background-color: rgba(0, 0, 0, 0.4) !important;
}

/* Sobrescrever qualquer z-index de modais do Bootstrap */
.modal {
    z-index: 1050 !important;
}

.modal-backdrop {
    z-index: 1040 !important;
}

/* Garantir que o SweetAlert2 tenha a maior prioridade */
body .swal2-container {
    z-index: 99999 !important;
}

/* Forçar exibição do SweetAlert2 mesmo em contextos problemáticos */
body .swal2-container.swal2-shown {
    display: flex !important;
    z-index: 99999 !important;
}

/* Específico para o popup */
body .swal2-popup.swal2-show {
    display: block !important;
    z-index: 100000 !important;
}
</style>

<!-- JavaScript -->

<script>
// Namespace para evitar conflitos com outros modais
window.VendasModal = window.VendasModal || {};

// Variáveis globais do modal vendas
VendasModal.ID_PEDIDO = <?php echo $id_pedido; ?>;
VendasModal.ID_CLIENTE = <?php echo $id_cliente; ?>;

// Função personalizada para SweetAlert2 com z-index forçado
VendasModal.showAlert = function(options) {
    console.log('🔔 VendasModal.showAlert - Opções recebidas:', options);
    
    const defaultOptions = {
        type: 'info', // Usar 'type' ao invés de 'icon' para versões antigas
        title: 'Informação',
        text: 'Mensagem',
        customClass: {
            container: 'swal2-container-vendas',
            popup: 'swal2-popup-vendas'
        },
        backdrop: true,
        allowOutsideClick: true,
        allowEscapeKey: true,
        allowEnterKey: true,
        showConfirmButton: true,
        confirmButtonText: 'OK',
        timer: false
    };
    
    // Converter 'icon' para 'type' se necessário
    if (options.icon) {
        options.type = options.icon;
        delete options.icon;
    }
    
    const finalOptions = Object.assign({}, defaultOptions, options);
    
    console.log('🔧 VendasModal.showAlert - Opções finais:', finalOptions);
    
    // Garantir z-index máximo
    const style = document.createElement('style');
    style.textContent = `
        .swal2-container-vendas {
            z-index: 99999 !important;
        }
        .swal2-popup-vendas {
            z-index: 100000 !important;
        }
    `;
    document.head.appendChild(style);
    
    // Usar Swal.fire diretamente e tratar o cleanup manualmente
    const sweetAlertPromise = Swal.fire(finalOptions);
    
    // Cleanup do style quando o modal fechar
    sweetAlertPromise.finally(() => {
        setTimeout(() => {
            if (style.parentNode) {
                style.parentNode.removeChild(style);
            }
        }, 100);
    });
    
    return sweetAlertPromise;
};

// Função para editar linha inline
VendasModal.editarInline = function(idVenda) {
    console.log('🔧 Iniciando edição inline para ID:', idVenda);
    console.log('🔍 Tipo do ID:', typeof idVenda);
    
    VendasModal.cancelarTodasEdicoes();
    
    // Tentar múltiplos métodos para encontrar a linha
    let linha = $(`tr[data-id="${idVenda}"]`);
    
    if (linha.length === 0) {
        console.log('🔍 Primeira tentativa falhou, tentando busca alternativa...');
        // Buscar por atributo data-id diretamente
        linha = $('#tabelaVendas tbody tr').filter(function() {
            return $(this).attr('data-id') == idVenda;
        });
    }
    
    if (linha.length === 0) {
        console.log('🔍 Segunda tentativa falhou, tentando busca via data()...');
        // Buscar usando data() jQuery
        linha = $('#tabelaVendas tbody tr').filter(function() {
            return $(this).data('id') == idVenda;
        });
    }
    
    if (linha.length === 0) {
        console.log('🔍 Terceira tentativa falhou, tentando busca pelo conteúdo da primeira célula...');
        // Buscar pelo conteúdo da primeira coluna (ID)
        linha = $('#tabelaVendas tbody tr').filter(function() {
            return $(this).find('td:first').text().trim() == idVenda;
        });
    }
    
    if (linha.length === 0) {
        console.error('❌ Linha não encontrada para ID:', idVenda);
        console.log('🔍 Debug - Linhas disponíveis na tabela:');
        $('#tabelaVendas tbody tr').each(function(index) {
            const $tr = $(this);
            console.log(`   Linha ${index}: data-id="${$tr.data('id')}", attr="${$tr.attr('data-id')}", texto="${$tr.find('td:first').text()}"`);
        });
        
        VendasModal.showAlert({ 
            type: 'error', 
            title: 'Erro!', 
            text: `Linha com ID ${idVenda} não encontrada. Tente recarregar a página.` 
        });
        return;
    }
    
    console.log('✅ Linha encontrada!', linha[0]);
    
    linha.addClass('editing-row');
    
    // Preparar o valor antes de mostrar o input
    const valorCell = linha.find('[data-field="valor"]');
    const valorOriginal = valorCell.data('original');
    const valorInput = valorCell.find('.valor-input');
    
    // Formatar o valor para exibição no input
    if (valorOriginal) {
        const valorFormatado = parseFloat(valorOriginal).toLocaleString('pt-BR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
        valorInput.val(valorFormatado);
        console.log('💰 Valor formatado para edição:', valorFormatado);
    }
    
    // Mostrar inputs e ocultar displays
    linha.find('.cell-display').hide();
    linha.find('.cell-input').show();
    
    // Mostrar/ocultar botões
    linha.find('.btn-edit, .btn-danger').hide();
    linha.find('.btn-save, .btn-cancel').show();
    
    // Aplicar máscara no campo valor
    VendasModal.aplicarMascaraValorInline(valorInput);
    
    // Focar no primeiro input
    linha.find('input').first().focus();
    
    console.log('✅ Edição inline ativada para ID:', idVenda);
}

// Função para cancelar edição
VendasModal.cancelarEdicaoInline = function(idVenda) {
    const linha = $(`tr[data-id="${idVenda}"]`);
    linha.removeClass('editing-row');
    
    // Restaurar valores originais
    linha.find('.editable-cell').each(function() {
        const campo = $(this).data('field');
        const valorOriginal = $(this).data('original');
        
        if (campo === 'valor') {
            // Usar o valor original sem formatação adicional
            $(this).find('.cell-input').val(valorOriginal);
        } else {
            $(this).find('.cell-input').val(valorOriginal);
        }
    });
    
    linha.find('.cell-input').hide();
    linha.find('.cell-display').show();
    linha.find('.btn-save, .btn-cancel').hide();
    linha.find('.btn-edit, .btn-danger').show();
}

// Função para salvar edição
VendasModal.salvarEdicaoInline = function(idVenda) {
    const linha = $(`tr[data-id="${idVenda}"]`);
    
    const produto = linha.find('[data-field="produto"] .cell-input').val();
    const valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    const data = linha.find('[data-field="data2"] .cell-input').val();
    
    const valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    if (!produto.trim() || !valorParaBanco || isNaN(parseFloat(valorParaBanco)) || !data) {
        VendasModal.showAlert({ type: 'error', title: 'Erro!', text: 'Todos os campos são obrigatórios e devem ser válidos.' });
        return;
    }
    
    fetch('vendas_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            acao: 'alterar',
            id_vendas: idVenda,
            produto: produto,
            valor: valorParaBanco,
            data: data,
            id_pedido: VendasModal.ID_PEDIDO,
            id_cliente: VendasModal.ID_CLIENTE
        })
    })
    .then(response => response.json())
    .then(responseData => {
        if (responseData.success) {
            // Atualizar displays
            linha.find('[data-field="produto"] .cell-display').text(produto);
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2, maximumFractionDigits: 2
            }));
            
            // Usar a data original que foi enviada para o servidor
            if (data) {
                const dataFormatada = VendasModal.formatarDataParaExibicao(data);
                linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            }
            
            // Atualizar dados originais
            linha.find('[data-field="produto"]').data('original', produto);
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Sair do modo de edição
            linha.removeClass('editing-row');
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            linha.find('.btn-save, .btn-cancel').hide();
            linha.find('.btn-edit, .btn-danger').show();
            
            VendasModal.showAlert({ type: 'success', title: 'Sucesso!', text: responseData.message, timer: 1500, showConfirmButton: false });
        } else {
            VendasModal.showAlert({ type: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('Erro:', error);
        VendasModal.showAlert({ type: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para formatar data para exibição
VendasModal.formatarDataParaExibicao = function(data) {
    if (!data || data === '0000-00-00' || data === '1970-01-01') {
        return 'Sem data';
    }
    
    // Se a data já está no formato dd/mm/yyyy, retorna como está
    if (/^\d{2}\/\d{2}\/\d{4}$/.test(data)) {
        return data;
    }
    
    // Se a data está no formato yyyy-mm-dd, converte para dd/mm/yyyy
    if (/^\d{4}-\d{2}-\d{2}$/.test(data)) {
        try {
            const [ano, mes, dia] = data.split('-');
            return `${dia}/${mes}/${ano}`;
        } catch (e) {
            return data;
        }
    }
    
    // Tenta usar Date() como último recurso
    try {
        const dateObj = new Date(data + 'T00:00:00');
        if (!isNaN(dateObj.getTime())) {
            return dateObj.toLocaleDateString('pt-BR');
        }
    } catch (e) {
        // Se falhar, retorna a data original
    }
    
    return data;
}

// Função de debug para verificar estado das linhas
VendasModal.debugLinhas = function() {
    console.log('🔍 === DEBUG LINHAS ===');
    console.log('🔢 Total de linhas na tabela:', $('#tabelaVendas tbody tr').length);
    
    $('#tabelaVendas tbody tr').each(function(index) {
        const $linha = $(this);
        const dataId = $linha.data('id');
        const attrId = $linha.attr('data-id');
        const isEditing = $linha.hasClass('editing-row');
        const editableCells = $linha.find('.editable-cell').length;
        
        console.log(`📋 Linha ${index}:`);
        console.log(`   data-id: "${dataId}" (tipo: ${typeof dataId})`);
        console.log(`   attr data-id: "${attrId}"`);
        console.log(`   em edição: ${isEditing}`);
        console.log(`   células editáveis: ${editableCells}`);
        console.log(`   HTML da linha:`, $linha[0]);
    });
    
    console.log('🔍 === FIM DEBUG ===');
}

// Função para testar edição forçada
VendasModal.testarEdicao = function(linha = 0) {
    console.log('🧪 === TESTE EDIÇÃO FORÇADA ===');
    const $linha = $('#tabelaVendas tbody tr').eq(linha);
    if ($linha.length === 0) {
        console.log('❌ Linha não encontrada');
        return;
    }
    
    const id = $linha.data('id') || $linha.attr('data-id');
    console.log('🎯 Tentando editar linha', linha, 'com ID:', id);
    console.log('🔍 Estrutura da linha:', $linha[0]);
    
    if (id && !String(id).startsWith('novo_') && String(id) !== 'Novo') {
        console.log('✅ ID válido, iniciando edição...');
        VendasModal.editarInline(id);
    } else {
        console.log('❌ ID inválido para edição:', id);
    }
}

// Função para testar clique simulado
VendasModal.testarClique = function(linha = 0) {
    console.log('🧪 === TESTE CLIQUE SIMULADO ===');
    const $linha = $('#tabelaVendas tbody tr').eq(linha);
    if ($linha.length === 0) {
        console.log('❌ Linha não encontrada');
        return;
    }
    
    const $celula = $linha.find('.editable-cell').first();
    if ($celula.length === 0) {
        console.log('❌ Célula editável não encontrada');
        return;
    }
    
    console.log('🎯 Simulando clique na célula...');
    console.log('🔍 Célula:', $celula[0]);
    $celula.trigger('click');
    console.log('✅ Clique simulado executado');
}

// Função para verificar e corrigir data-ids após criação
VendasModal.corrigirDataIds = function() {
    console.log('🔧 === CORREÇÃO DE DATA-IDS ===');
    
    $('#tabelaVendas tbody tr').each(function(index) {
        const $linha = $(this);
        const primeiraColuna = $linha.find('td:first').text().trim();
        const dataId = $linha.data('id');
        const attrId = $linha.attr('data-id');
        
        // Se a primeira coluna tem um ID numérico mas os data-ids não estão corretos
        if (!isNaN(primeiraColuna) && primeiraColuna !== '' && primeiraColuna !== 'Novo') {
            if (dataId != primeiraColuna || attrId != primeiraColuna) {
                console.log(`🔧 Corrigindo linha ${index}: ID da coluna="${primeiraColuna}", data-id="${dataId}", attr="${attrId}"`);
                $linha.attr('data-id', primeiraColuna);
                $linha.data('id', parseInt(primeiraColuna));
                console.log(`✅ Linha ${index} corrigida!`);
            }
        }
    });
    
    console.log('🔧 === FIM CORREÇÃO ===');
}

// Função para verificar estado completo da tabela
VendasModal.verificarEstadoTabela = function() {
    console.log('🔍 === VERIFICAÇÃO COMPLETA DA TABELA ===');
    
    const totalLinhas = $('#tabelaVendas tbody tr').length;
    console.log('📊 Total de linhas:', totalLinhas);
    
    $('#tabelaVendas tbody tr').each(function(index) {
        const $linha = $(this);
        const dataId = $linha.data('id');
        const attrId = $linha.attr('data-id');
        const celulasEditaveis = $linha.find('.editable-cell').length;
        const isEditing = $linha.hasClass('editing-row');
        
        console.log(`📋 Linha ${index}:`);
        console.log(`   data-id: "${dataId}" (tipo: ${typeof dataId})`);
        console.log(`   attr data-id: "${attrId}"`);
        console.log(`   células editáveis: ${celulasEditaveis}`);
        console.log(`   em edição: ${isEditing}`);
        console.log(`   HTML:`, $linha[0]);
    });
    
    // Verificar eventos
    const eventos = $._data($('#tabelaVendas')[0], 'events');
    console.log('🎯 Eventos na tabela:', eventos);
    
    console.log('🔍 === FIM VERIFICAÇÃO ===');
}

// Função para cancelar todas as edições
VendasModal.cancelarTodasEdicoes = function() {
    $('.editing-row').each(function() {
        const idVenda = $(this).data('id');
        if (typeof idVenda === 'string' && idVenda.startsWith('novo_')) {
            $(this).remove();
        } else {
            VendasModal.cancelarEdicaoInline(idVenda);
        }
    });
}

// Função para aplicar máscara de valor
VendasModal.aplicarMascaraValorInline = function(campo) {
    campo.off('.mascaraInline');
    
    campo.on('input.mascaraInline', function() {
        let valor = this.value.replace(/\D/g, '');
        
        if (valor.length === 0) {
            this.value = '';
            return;
        }
        
        // Remover zeros à esquerda desnecessários
        valor = valor.replace(/^0+/, '');
        if (valor === '') valor = '0';
        
        // Garantir pelo menos 2 dígitos para os centavos
        if (valor.length === 1) {
            valor = '0' + valor;
        }
        
        // Separar a parte inteira dos centavos
        const centavos = valor.slice(-2);
        let parteInteira = valor.slice(0, -2);
        
        // Se não há parte inteira, é zero
        if (parteInteira === '') {
            parteInteira = '0';
        }
        
        // Adicionar pontos para milhares
        parteInteira = parteInteira.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        
        // Formatar como valor monetário brasileiro
        this.value = parteInteira + ',' + centavos;
    });
    
    campo.on('focus.mascaraInline', function() {
        setTimeout(() => this.select(), 10);
    });
    
    campo.on('blur.mascaraInline', function() {
        let valor = this.value.replace(/\D/g, '');
        if (valor === '' || valor === '0' || valor === '00') {
            this.value = '0,00';
        }
    });
}

// Função para configurar eventos de edição em uma linha específica
VendasModal.configurarEventosEdicao = function($linha) {
    // Remover eventos anteriores para evitar duplicação
    $linha.find('.editable-cell').off('click.edicaoInline');
    
    // Adicionar evento de clique para edição inline
    $linha.find('.editable-cell').on('click.edicaoInline', function(e) {
        // Verificar se não está clicando em um input já aberto
        if ($(e.target).is('input') || $(e.target).closest('input').length > 0) {
            return;
        }
        
        // Verificar se a linha já está em modo de edição
        if ($linha.hasClass('editing-row')) {
            return;
        }
        
        // Obter o ID da venda
        const idVenda = $linha.data('id');
        
        // Verificar se é uma linha válida (não é nova linha)
        if (idVenda && !String(idVenda).startsWith('novo_')) {
            VendasModal.editarInline(idVenda);
        }
    });
    
    console.log('✅ Eventos de edição configurados para linha ID:', $linha.data('id'));
}

// LIMPAR COMPLETAMENTE QUALQUER JAVASCRIPT ANTERIOR
if (typeof window.FinanceiroClientesModal !== 'undefined') {
    console.log('🧹 REMOVENDO FinanceiroClientesModal anterior...');
    delete window.FinanceiroClientesModal;
}

// Remover qualquer event listener anterior
$(document).off('.financeiro');
$(document).off('.vendas');

console.log('🔥 INICIANDO VENDAS_MODAL.PHP - JAVASCRIPT LIMPO');

// Função para inicializar/recriar DataTable
VendasModal.criarDataTable = function() {
    console.log('🔧 Criando/Recriando DataTable...');
    
    // Destruir DataTable existente se houver
    if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaVendas')) {
        $('#tabelaVendas').DataTable().destroy();
    }
    
    const table = $('#tabelaVendas').DataTable({
        "language": {
            "url": "datatable/pt-BR.json"
        },
        "pageLength": 25,
        "lengthMenu": [[10, 25, 50, 100, -1], [10, 25, 50, 100, "Todos"]],
        "responsive": true,
        "processing": false,
        "info": true,
        "paging": true,
        "order": [[0, 'desc']], // Ordenar por ID (primeira coluna) em ordem decrescente
        "ordering": true,
        "autoWidth": false,
        "columnDefs": [
            {
                "targets": "_all",
                "defaultContent": "",
                "orderable": true
            },
            {
                "targets": 4, // Coluna de ações
                "orderable": false
            }
        ],
        "search": {
            "regex": false,
            "caseInsensitive": true
        },
        "initComplete": function () {
            console.log('🚀 DataTable Vendas (re)inicializado');
            console.log('📊 TOTAL DE LINHAS CARREGADAS:', this.api().rows().count());
            console.log('📋 LINHAS VISÍVEIS NA PÁGINA:', this.api().rows({page: 'current'}).count());
            VendasModal.setupFooterSearchVendas(this.api());
        },
        "drawCallback": function (settings) {
            console.log('📊 DataTable DRAW CALLBACK executado');
            console.log('🔍 Linhas na tabela após draw:', $('#tabelaVendas tbody tr').length);
            
            // Log de todas as linhas e seus data-ids
            $('#tabelaVendas tbody tr').each(function(index) {
                const dataId = $(this).data('id');
                const attrId = $(this).attr('data-id');
                console.log(`📋 Linha ${index}: data-id='${dataId}', attr='${attrId}'`);
            });
            
            // Corrigir data-ids que possam estar incorretos
            setTimeout(() => {
                VendasModal.corrigirDataIds();
            }, 100);
            
            // Garantir que todas as linhas tenham os eventos corretos
            // Isso é necessário porque o DataTable pode recriar o DOM
            setTimeout(() => {
                $('#tabelaVendas tbody tr').each(function() {
                    const $linha = $(this);
                    const idVenda = $linha.data('id') || $linha.attr('data-id');
                    
                    // Verificar se a linha tem ID válido e não é temporária
                    if (idVenda && !String(idVenda).startsWith('novo_') && String(idVenda) !== 'Novo') {
                        // Remover eventos antigos para evitar duplicação
                        $linha.find('.editable-cell').off('click.rowSpecific');
                        
                        // Adicionar evento específico para esta linha
                        $linha.find('.editable-cell').on('click.rowSpecific', function(e) {
                            if ($(e.target).is('input') || $(e.target).closest('input').length > 0) {
                                return;
                            }
                            
                            if ($linha.hasClass('editing-row')) {
                                return;
                            }
                            
                            console.log('🎯 DRAW CALLBACK - Clique detectado na linha ID:', idVenda);
                            VendasModal.editarInline(idVenda);
                        });
                        
                        console.log('🔄 Eventos reaplicados para linha ID:', idVenda);
                    }
                });
            }, 200);
            
            // Reinicializar tooltips nas novas linhas
            $('[data-toggle="tooltip"]').tooltip('dispose').tooltip({
                delay: { show: 300, hide: 100 }
            });
            
            // Verificar se os inputs ainda existem após redraw
            var $inputs = $('.footer-search-input-vendas');
            if ($inputs.length === 0) {
                console.log('⚠️ DRAW CALLBACK: Inputs perdidos após redraw, recriando...');
                setTimeout(function() {
                    VendasModal.criarInputsVendasManualmente();
                }, 200);
            }
        }
    });
    
    return table;
}

// Função para configurar busca no footer
VendasModal.setupFooterSearchVendas = function(table) {
    setTimeout(function() {
        console.log('🔧 SETUP FOOTER SEARCH VENDAS - Iniciando...');
        
        // Verificar se já existem inputs criados
        var $inputsExistentes = $('.footer-search-input-vendas');
        if ($inputsExistentes.length > 0) {
            console.log('✅ Inputs já criados, pulando criação');
            return;
        }
        
        // Tentar método automático primeiro
        var inputsCriados = 0;
        try {
            table.columns().every(function (index) {
                var column = this;
                var title = $(column.footer()).text().trim();
                
                // Pular coluna de Ações (última coluna)
                if (index === 4) {
                    $(column.footer()).html('');
                    return;
                }
                
                if (title) {
                    try {
                        // Criar input com wrapper
                        var inputHtml = '<div class="md-input-wrapper"><input type="text" class="md-form-control footer-search-input-vendas" placeholder="' + title.replace(/'/g, '&#39;').replace(/"/g, '&quot;') + '" data-column="' + index + '" /></div>';
                        $(column.footer()).html(inputHtml);
                        inputsCriados++;
                        
                        // Configurar evento de busca com throttling
                        var searchTimeout;
                        $('input', column.footer()).on('keyup change clear', function () {
                            var searchValue = this.value;
                            
                            // Limpar timeout anterior
                            clearTimeout(searchTimeout);
                            
                            searchTimeout = setTimeout(function() {
                                if (column.search() !== searchValue) {
                                    column.search(searchValue, false, false).draw();
                                }
                            }, 300);
                        });
                        
                        // Prevenir ordenação ao clicar no input
                        $('input', column.footer()).on('click', function(e) {
                            e.stopPropagation();
                        });
                        
                    } catch (error) {
                        console.log('❌ Erro ao criar input para coluna', index, ':', error);
                    }
                }
            });
        } catch (error) {
            console.log('❌ Erro no método automático:', error);
            inputsCriados = 0;
        }
        
        // Se não conseguiu criar automaticamente, forçar criação manual
        if (inputsCriados === 0) {
            console.log('⚠️ Método automático falhou, forçando criação manual...');
            VendasModal.criarInputsVendasManualmente();
        } else {
            console.log('✅ Setup concluído com sucesso! Inputs criados:', inputsCriados);
        }
    }, 300);
}

// Função para criar inputs manualmente
VendasModal.criarInputsVendasManualmente = function() {
    console.log('🔧 CRIAÇÃO MANUAL VENDAS - Iniciando...');
    
    // Definir cabeçalhos das colunas
    var colunas = [
        { titulo: 'Nº', placeholder: 'Nº' },
        { titulo: 'Produto', placeholder: 'Produto' },
        { titulo: 'Valor', placeholder: 'Valor' },
        { titulo: 'Data', placeholder: 'Data' },
        { titulo: 'Ações', placeholder: '' }
    ];
    
    // Verificar se footer existe
    var $footer = $('#tabelaVendas tfoot');
    if ($footer.length === 0) {
        console.log('❌ Footer não encontrado, criando...');
        $('#tabelaVendas').append('<tfoot><tr>' + '<th></th>'.repeat(5) + '</tr></tfoot>');
        $footer = $('#tabelaVendas tfoot');
    }
    
    var $footerCells = $footer.find('th');
    
    // Criar inputs para cada coluna (exceto a última - Ações)
    for (var i = 0; i < colunas.length - 1; i++) {
        var coluna = colunas[i];
        var $cell = $footerCells.eq(i);
        
        if ($cell.length > 0) {
            // Criar input com wrapper
            var inputHtml = '<div class="md-input-wrapper">' +
                '<input type="text" class="md-form-control footer-search-input-vendas" ' +
                'placeholder="' + coluna.placeholder + '" data-column="' + i + '" />' +
                '</div>';
            
            $cell.html(inputHtml);
            
            // Configurar evento de busca com throttling
            var $input = $cell.find('.footer-search-input-vendas');
            (function(columnIndex) {
                $input.on('keyup change clear', function() {
                    var searchValue = this.value;
                    
                    // Obter referência do DataTable atual
                    if ($.fn.DataTable.isDataTable('#tabelaVendas')) {
                        const currentTable = $('#tabelaVendas').DataTable();
                        
                        // Aplicar filtro na coluna com throttling
                        clearTimeout(window['timeout_vendas_col_' + columnIndex]);
                        window['timeout_vendas_col_' + columnIndex] = setTimeout(function() {
                            if (currentTable.column(columnIndex).search() !== searchValue) {
                                currentTable.column(columnIndex).search(searchValue, false, false).draw();
                            }
                        }, 300);
                    }
                });
                
                // Prevenir ordenação ao clicar no input
                $input.on('click', function(e) {
                    e.stopPropagation();
                });
            })(i);
        }
    }
    
    // Limpar última coluna (Ações)
    if ($footerCells.length > 4) {
        $footerCells.eq(4).html('');
    }
    
    console.log('🔧 CRIAÇÃO MANUAL VENDAS - Concluída!');
}

// Função para adicionar nova linha
VendasModal.adicionarNovaLinha = function() {
    VendasModal.cancelarTodasEdicoes();
    
    // Se a tabela está oculta (sem vendas), mostrar ela
    const $tabelaContainer = $('.table-responsive');
    const $mensagemVazia = $('.text-center.text-muted');
    
    if ($tabelaContainer.is(':hidden')) {
        console.log('📋 Tabela estava oculta, exibindo para primeira venda...');
        $mensagemVazia.hide();
        $tabelaContainer.show();
    }
    
    const novoId = 'novo_' + Date.now();
    const novaLinha = `
        <tr data-id="${novoId}" class="editing-row nova-linha">
            <td>Novo</td>
            <td class="editable-cell" data-field="produto" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="text" class="form-control form-control-sm cell-input" placeholder="Digite o produto" style="display: block;">
            </td>
            <td class="editable-cell" data-field="valor" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="text" class="form-control form-control-sm cell-input valor-input" placeholder="0,00" style="display: block;">
            </td>
            <td class="editable-cell" data-field="data2" data-original="">
                <span class="cell-display" style="display: none;"></span>
                <input type="date" class="form-control form-control-sm cell-input" value="${new Date().toISOString().split('T')[0]}" style="display: block;">
            </td>
            <td>
                <button type="button" class="btn btn-success btn-sm btn-save" onclick="console.log('🎯 BOTÃO VENDAS CLICADO!'); VendasModal.salvarNovaVenda('${novoId}')" title="Salvar">
                    <i class="fas fa-save"></i>
                </button>
                <button type="button" class="btn btn-secondary btn-sm btn-cancel ms-1" onclick="VendasModal.cancelarNovaVenda('${novoId}')" title="Cancelar">
                    <i class="fas fa-times"></i>
                </button>
            </td>
        </tr>
    `;
    
    $('#tabelaVendas tbody').prepend(novaLinha);
    
    const valorInput = $(`tr[data-id="${novoId}"] .valor-input`);
    VendasModal.aplicarMascaraValorInline(valorInput);
    
    // Focar no primeiro input (produto)
    $(`tr[data-id="${novoId}"] input`).first().focus();
    
    console.log('✅ Nova linha adicionada para venda');
}

// Função para salvar nova venda
VendasModal.salvarNovaVenda = function(novoId) {
    console.log('🎯 VENDAS_MODAL.PHP - FUNÇÃO salvarNovaVenda EXECUTADA!');
    console.log('🔢 TIMESTAMP ÚNICO:', Date.now());
    console.log('📁 MÓDULO ATIVO: VENDAS');
    
    const linha = $(`tr[data-id="${novoId}"]`);
    
    const produto = linha.find('[data-field="produto"] .cell-input').val();
    const valorFormatado = linha.find('[data-field="valor"] .cell-input').val();
    const data = linha.find('[data-field="data2"] .cell-input').val();
    
    const valorParaBanco = valorFormatado.replace(/\./g, '').replace(',', '.');
    
    if (!produto.trim() || !valorParaBanco || isNaN(parseFloat(valorParaBanco)) || !data) {
        VendasModal.showAlert({ type: 'error', title: 'Erro!', text: 'Todos os campos são obrigatórios.' });
        return;
    }
    
    console.log('💾 Salvando nova venda:', { produto, valor: valorParaBanco, data });
    
    // LOGS DE DEBUG EXTREMO
    const urlAlvo = 'vendas_modal.php'; // RESTAURADO PARA O CORRETO
    console.log('🎯 URL ALVO DEFINIDO:', urlAlvo);
    console.log('🔗 URL ABSOLUTA:', window.location.origin + window.location.pathname.replace(/[^/]*$/, '') + urlAlvo);
    console.log('📍 SCRIPT ATUAL:', window.location.pathname);
    console.log('📁 DIRETÓRIO ATUAL:', window.location.pathname.replace(/[^/]*$/, ''));
    
    const dadosParaEnvio = {
        acao: 'inserir',
        produto: produto,
        valor: valorParaBanco,
        data: data,
        id_pedido: VendasModal.ID_PEDIDO,
        id_cliente: VendasModal.ID_CLIENTE
    };
    
    console.log('📦 DADOS PARA ENVIO:', dadosParaEnvio);
    console.log('🚀 INICIANDO FETCH PARA:', urlAlvo);
    
    fetch(urlAlvo, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams(dadosParaEnvio)
    })
    .then(response => {
        console.log('📡 RESPOSTA FETCH RECEBIDA:', response);
        console.log('📊 STATUS:', response.status, response.statusText);
        console.log('🌐 URL FINAL:', response.url);
        return response.json();
    })
    .then(responseData => {
        console.log('📥 Resposta do servidor:', responseData);
        
        if (responseData.success) {
            const novoIdVenda = responseData.id_vendas || 'novo';
            linha.attr('data-id', novoIdVenda);
            linha.removeClass('nova-linha');
            
            // Atualizar displays
            linha.find('[data-field="produto"] .cell-display').text(produto);
            linha.find('[data-field="valor"] .cell-display').text('R$ ' + parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                minimumFractionDigits: 2, maximumFractionDigits: 2
            }));
            
            // Usar a data original que foi enviada para o servidor
            if (data) {
                const dataFormatada = VendasModal.formatarDataParaExibicao(data);
                linha.find('[data-field="data2"] .cell-display').text(dataFormatada);
            }
            
            // Atualizar dados originais
            linha.find('[data-field="produto"]').data('original', produto);
            linha.find('[data-field="valor"]').data('original', valorParaBanco);
            linha.find('[data-field="data2"]').data('original', data);
            
            // Sair do modo de edição
            linha.find('.cell-input').hide();
            linha.find('.cell-display').show();
            linha.removeClass('editing-row');
            
            // Atualizar ID da linha na coluna Nº
            linha.find('td:first').text(novoIdVenda);
            
            // Atualizar botões
            linha.find('td:last').html(`
                <button type="button" class="btn btn-warning btn-sm btn-edit" onclick="VendasModal.editarInline(${novoIdVenda})" title="Editar">
                    <i class="fas fa-edit"></i>
                </button>
                <button type="button" class="btn btn-danger btn-sm ms-1" onclick="VendasModal.excluirVenda(${novoIdVenda})" title="Excluir">
                    <i class="fas fa-trash"></i>
                </button>
            `);
            
            // Atualizar total no badge
            VendasModal.atualizarTotalVendas();
            
            // Se DataTable estiver ativo, usar método seguro sem recriar
            if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaVendas')) {
                const dataTable = $('#tabelaVendas').DataTable();
                
                // Verificar se a tabela estava vazia antes
                const tabelaEstavVazia = dataTable.rows().count() === 0;
                
                // Criar dados no formato correto para o DataTable
                const dadosNovaLinha = [
                    novoIdVenda,
                    `<div class="editable-cell" data-field="produto" data-original="${produto}">
                        <span class="cell-display">${produto}</span>
                        <input type="text" class="cell-input form-control form-control-sm" value="${produto}" style="display: none;">
                    </div>`,
                    `<div class="editable-cell" data-field="valor" data-original="${valorParaBanco}">
                        <span class="cell-display">R$ ${parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                            minimumFractionDigits: 2, maximumFractionDigits: 2
                        })}</span>
                        <input type="text" class="cell-input form-control form-control-sm valor-input" value="${valorParaBanco}" style="display: none;">
                    </div>`,
                    `<div class="editable-cell" data-field="data2" data-original="${data}">
                        <span class="cell-display">${VendasModal.formatarDataParaExibicao(data)}</span>
                        <input type="date" class="cell-input form-control form-control-sm" value="${data}" style="display: none;">
                    </div>`,
                    `<button type="button" class="btn btn-sm btn-warning btn-edit" onclick="VendasModal.editarInline(${novoIdVenda})" title="Editar">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-danger ms-1" onclick="VendasModal.excluirVenda(${novoIdVenda})" title="Excluir">
                        <i class="fas fa-trash"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-success btn-save" onclick="VendasModal.salvarEdicaoInline(${novoIdVenda})" title="Salvar" style="display: none;">
                        <i class="fas fa-save"></i>
                    </button>
                    <button type="button" class="btn btn-sm btn-secondary btn-cancel ms-1" onclick="VendasModal.cancelarEdicaoInline(${novoIdVenda})" title="Cancelar" style="display: none;">
                        <i class="fas fa-times"></i>
                    </button>`
                ];
                
                // Remover a linha temporária primeiro
                linha.remove();
                
                // Adicionar nova linha usando a API do DataTable
                const novaLinha = dataTable.row.add(dadosNovaLinha);
                
                // Definir o data-id na linha recém-criada
                const $novaLinhaDom = $(novaLinha.node());
                $novaLinhaDom.attr('data-id', novoIdVenda);
                $novaLinhaDom.data('id', novoIdVenda);
                
                console.log('🆔 Data-ID definido na nova linha:', novoIdVenda);
                console.log('🔍 Verificação do data-id:', $novaLinhaDom.data('id'));
                console.log('🔍 Verificação do attr data-id:', $novaLinhaDom.attr('data-id'));
                
                // Debug detalhado da estrutura da linha
                console.log('🔍 Estrutura completa da linha:', $novaLinhaDom[0]);
                console.log('🎯 Células editáveis na linha:', $novaLinhaDom.find('.editable-cell').length);
                
                // Forçar ordenação decrescente por ID para colocar a nova linha no topo
                dataTable.order([0, 'desc']).draw(false);
                
                // Aplicar máscara no campo valor da nova linha
                setTimeout(() => {
                    const $novaLinhaDom = $(novaLinha.node());
                    VendasModal.aplicarMascaraValorInline($novaLinhaDom.find('.valor-input'));
                    
                    console.log('✅ Configuração aplicada à nova linha DataTable ID:', novoIdVenda);
                    console.log('🔍 Estrutura da nova linha:', $novaLinhaDom[0]);
                    console.log('🎯 Células editáveis encontradas:', $novaLinhaDom.find('.editable-cell').length);
                    
                    // Forçar atualização do data-id via jQuery também
                    $novaLinhaDom.data('id', novoIdVenda);
                    
                    // Debug adicional
                    console.log('🔄 Data-ID após atualização jQuery:', $novaLinhaDom.data('id'));
                    
                    // Teste automático de funcionalidade após criação
                    setTimeout(() => {
                        console.log('🧪 === TESTE AUTOMÁTICO DE EDIÇÃO ===');
                        
                        // Verificar se a linha pode ser encontrada pelos métodos usuais
                        const linhaParaTeste1 = $(`#tabelaVendas tbody tr[data-id="${novoIdVenda}"]`);
                        const linhaParaTeste2 = $('#tabelaVendas tbody tr').filter(function() {
                            return $(this).data('id') == novoIdVenda;
                        });
                        const linhaParaTeste3 = $('#tabelaVendas tbody tr').filter(function() {
                            return $(this).find('td:first').text().trim() == novoIdVenda;
                        });
                        
                        console.log('🔍 Linha encontrada via seletor attr:', linhaParaTeste1.length);
                        console.log('🔍 Linha encontrada via data():', linhaParaTeste2.length);
                        console.log('🔍 Linha encontrada via texto:', linhaParaTeste3.length);
                        
                        if (linhaParaTeste1.length > 0 || linhaParaTeste2.length > 0 || linhaParaTeste3.length > 0) {
                            console.log('✅ Linha pronta para edição manual!');
                        } else {
                            console.log('❌ PROBLEMA: Linha não está acessível pelos métodos de busca!');
                            console.log('🔧 Tentando corrigir o data-id...');
                            
                            // Forçar correção do data-id
                            const todasAsLinhas = $('#tabelaVendas tbody tr');
                            todasAsLinhas.each(function(index) {
                                const $linha = $(this);
                                const primeiraColuna = $linha.find('td:first').text().trim();
                                if (primeiraColuna == novoIdVenda) {
                                    console.log('🔧 Corrigindo data-id da linha encontrada...');
                                    $linha.attr('data-id', novoIdVenda);
                                    $linha.data('id', novoIdVenda);
                                    console.log('✅ Data-ID corrigido!');
                                }
                            });
                        }
                        
                        console.log('🧪 === FIM TESTE AUTOMÁTICO ===');
                    }, 1000);
                }, 200);
                
                // Ir para a primeira página se necessário
                dataTable.page('first').draw(false);
                
                console.log('📋 DataTable: Nova linha adicionada usando row.add() seguro');
                
                // Se a tabela estava vazia, garantir que ela seja exibida
                if (tabelaEstavVazia) {
                    $('.text-center.text-muted').hide();
                    $('.table-responsive').show();
                    console.log('📋 Tabela estava vazia, agora exibida com primeira venda');
                }
            } else {
                // Fallback para quando DataTable não está ativo
                console.log('⚠️ DataTable não ativo, usando método fallback');
                
                // Criar nova linha HTML completa na primeira posição do tbody
                const novaLinhaHTML = `
                    <tr data-id="${novoIdVenda}">
                        <td>${novoIdVenda}</td>
                        <td class="editable-cell" data-field="produto" data-original="${produto}">
                            <span class="cell-display">${produto}</span>
                            <input type="text" class="cell-input form-control form-control-sm" value="${produto}" style="display: none;">
                        </td>
                        <td class="editable-cell" data-field="valor" data-original="${valorParaBanco}">
                            <span class="cell-display">R$ ${parseFloat(valorParaBanco).toLocaleString('pt-BR', {
                                minimumFractionDigits: 2, maximumFractionDigits: 2
                            })}</span>
                            <input type="text" class="cell-input form-control form-control-sm valor-input" value="${valorParaBanco}" style="display: none;">
                        </td>
                        <td class="editable-cell" data-field="data2" data-original="${data}">
                            <span class="cell-display">${VendasModal.formatarDataParaExibicao(data)}</span>
                            <input type="date" class="cell-input form-control form-control-sm" value="${data}" style="display: none;">
                        </td>
                        <td>
                            <button type="button" class="btn btn-sm btn-warning btn-edit" onclick="VendasModal.editarInline(${novoIdVenda})" title="Editar">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-danger ms-1" onclick="VendasModal.excluirVenda(${novoIdVenda})" title="Excluir">
                                <i class="fas fa-trash"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-success btn-save" onclick="VendasModal.salvarEdicaoInline(${novoIdVenda})" title="Salvar" style="display: none;">
                                <i class="fas fa-save"></i>
                            </button>
                            <button type="button" class="btn btn-sm btn-secondary btn-cancel ms-1" onclick="VendasModal.cancelarEdicaoInline(${novoIdVenda})" title="Cancelar" style="display: none;">
                                <i class="fas fa-times"></i>
                            </button>
                        </td>
                    </tr>
                `;
                
                // Inserir no início do tbody
                $('#tabelaVendas tbody').prepend(novaLinhaHTML);
                
                // Aplicar máscara no campo valor da nova linha
                setTimeout(() => {
                    VendasModal.aplicarMascaraValorInline($(`tr[data-id="${novoIdVenda}"] .valor-input`));
                    console.log('✅ Configuração aplicada à nova linha fallback ID:', novoIdVenda);
                }, 50);
            }
            
            VendasModal.showAlert({ 
                type: 'success', 
                title: 'Sucesso!', 
                text: 'Venda adicionada com sucesso!', 
                timer: 1500, 
                showConfirmButton: false 
            });
            
            console.log('✅ Venda salva com sucesso!');
        } else {
            console.error('❌ Erro do servidor:', responseData.message);
            VendasModal.showAlert({ type: 'error', title: 'Erro!', text: responseData.message });
        }
    })
    .catch(error => {
        console.error('❌ Erro na requisição:', error);
        VendasModal.showAlert({ type: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
    });
}

// Função para atualizar o total de vendas no badge
VendasModal.atualizarTotalVendas = function() {
    let total = 0;
    
    // Somar todas as vendas visíveis na tabela
    $('#tabelaVendas tbody tr:not(.nova-linha)').each(function() {
        const valorOriginal = $(this).find('[data-field="valor"]').data('original');
        if (valorOriginal && !isNaN(parseFloat(valorOriginal))) {
            total += parseFloat(valorOriginal);
        }
    });
    
    // Atualizar o badge
    $('.badge.bg-primary').text('Total: R$ ' + total.toLocaleString('pt-BR', {
        minimumFractionDigits: 2, 
        maximumFractionDigits: 2
    }));
    
    console.log('💰 Total atualizado:', total);
}

// Função para cancelar nova venda
VendasModal.cancelarNovaVenda = function(novoId) {
    $(`tr[data-id="${novoId}"]`).remove();
    
    // Se não há mais vendas na tabela, ocultar a tabela e mostrar mensagem
    const linhasRestantes = $('#tabelaVendas tbody tr').length;
    console.log('📊 Linhas restantes na tabela:', linhasRestantes);
    
    if (linhasRestantes === 0) {
        console.log('🙈 Nenhuma venda restante, ocultando tabela...');
        $('.table-responsive').hide();
        $('.text-center.text-muted').show();
    }
}

// Função para excluir venda
VendasModal.excluirVenda = function(idVendas) {
    console.log('🗑️ VENDAS_MODAL.JS - Função excluirVenda chamada - ID:', idVendas);
    console.log('🔍 TIMESTAMP:', Date.now());
    
    // Verificar se SweetAlert2 está disponível
    if (typeof Swal === 'undefined') {
        console.error('❌ SweetAlert2 não está carregado!');
        if (confirm('Deseja realmente excluir esta venda?')) {
            VendasModal.processarExclusao(idVendas);
        }
        return;
    }
    
    // Usar SweetAlert2 compatível com versão antiga
    Swal.fire({
        title: 'Tem certeza?',
        text: 'Esta ação não pode ser desfeita!',
        type: 'warning', // Usar 'type' ao invés de 'icon' para versões antigas
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Sim, excluir!',
        cancelButtonText: 'Cancelar',
        allowOutsideClick: false,
        allowEscapeKey: true
    }).then((result) => {
        console.log('🔔 SweetAlert2 - Resultado completo:', result);
        console.log('🔍 isConfirmed:', result.isConfirmed);
        console.log('🔍 value:', result.value);
        console.log('🔍 dismiss:', result.dismiss);
        
        // Verificar tanto isConfirmed (versão nova) quanto value (versão antiga)
        if (result.isConfirmed === true || result.value === true) {
            console.log('✅ Usuário confirmou exclusão, processando...');
            VendasModal.processarExclusao(idVendas);
        } else {
            console.log('❌ Usuário cancelou exclusão');
        }
    }).catch(error => {
        console.error('❌ Erro no SweetAlert2:', error);
        if (confirm('Erro no sistema de confirmação. Deseja excluir mesmo assim?')) {
            VendasModal.processarExclusao(idVendas);
        }
    });
}

// Função separada para processar a exclusão
VendasModal.processarExclusao = function(idVendas) {
    console.log('🚀 Processando exclusão do ID:', idVendas);
    
    const dados = {
        acao: 'excluir',
        id_vendas: idVendas,
        id_pedido: VendasModal.ID_PEDIDO,
        id_cliente: VendasModal.ID_CLIENTE
    };
    
    console.log('📦 Dados a enviar:', dados);
    
    fetch('vendas_modal.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams(dados)
    })
    .then(response => {
        console.log('📡 Resposta recebida:', response);
        console.log('📊 Status:', response.status, response.statusText);
        console.log('🌐 URL:', response.url);
        return response.json();
    })
    .then(data => {
        console.log('📥 Dados da resposta:', data);
        
        if (data.success) {
            console.log('✅ Exclusão bem-sucedida, removendo linha da tabela...');
            
            const $linha = $(`tr[data-id="${idVendas}"]`);
            console.log('🔍 Linha encontrada:', $linha.length > 0 ? 'SIM' : 'NÃO');
            console.log('🔍 Seletor usado:', `tr[data-id="${idVendas}"]`);
            
            if ($linha.length > 0) {
                // Se DataTable estiver ativo, remover via DataTable
                if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaVendas')) {
                    const dataTable = $('#tabelaVendas').DataTable();
                    const row = dataTable.row($linha);
                    row.remove().draw();
                    console.log('🗑️ Linha removida via DataTable');
                } else {
                    $linha.remove();
                    console.log('🗑️ Linha removida diretamente do DOM');
                }
                
                // Atualizar o total após exclusão
                VendasModal.atualizarTotalVendas();
                
                // Verificar se ainda há vendas na tabela
                let linhasRestantes;
                if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaVendas')) {
                    const dataTable = $('#tabelaVendas').DataTable();
                    linhasRestantes = dataTable.rows().count();
                } else {
                    linhasRestantes = $('#tabelaVendas tbody tr:not(.nova-linha)').length;
                }
                console.log('📊 Linhas restantes:', linhasRestantes);
                
                if (linhasRestantes === 0) {
                    console.log('🙈 Nenhuma venda restante, ocultando tabela...');
                    $('.table-responsive').hide();
                    $('.text-center.text-muted').show();
                }
                
                // Mostrar sucesso - usar type ao invés de icon
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        type: 'success', 
                        title: 'Excluído!', 
                        text: data.message, 
                        timer: 1500, 
                        showConfirmButton: false
                    });
                } else {
                    alert('Venda excluída com sucesso!');
                }
            } else {
                console.error('❌ Linha não encontrada na tabela!');
                // Tentar encontrar usando jQuery diferente
                const $linhaAlternativa = $('#tabelaVendas tbody tr').filter(function() {
                    return $(this).data('id') == idVendas;
                });
                console.log('🔍 Tentativa alternativa encontrou:', $linhaAlternativa.length, 'linhas');
                
                if ($linhaAlternativa.length > 0) {
                    // Se DataTable estiver ativo, remover via DataTable
                    if ($.fn.DataTable && $.fn.DataTable.isDataTable('#tabelaVendas')) {
                        const dataTable = $('#tabelaVendas').DataTable();
                        const row = dataTable.row($linhaAlternativa);
                        row.remove().draw();
                        console.log('🗑️ Linha removida via DataTable (método alternativo)');
                    } else {
                        $linhaAlternativa.remove();
                        console.log('🗑️ Linha removida via método alternativo');
                    }
                    VendasModal.atualizarTotalVendas();
                    
                    if (typeof Swal !== 'undefined') {
                        Swal.fire({
                            type: 'success', 
                            title: 'Excluído!', 
                            text: data.message, 
                            timer: 1500, 
                            showConfirmButton: false
                        });
                    } else {
                        alert('Venda excluída com sucesso!');
                    }
                }
            }
        } else {
            console.error('❌ Erro na exclusão:', data.message);
            if (typeof Swal !== 'undefined') {
                Swal.fire({ type: 'error', title: 'Erro!', text: data.message });
            } else {
                alert('Erro: ' + data.message);
            }
        }
    })
    .catch(error => {
        console.error('❌ Erro na requisição:', error);
        if (typeof Swal !== 'undefined') {
            Swal.fire({ type: 'error', title: 'Erro!', text: 'Erro de comunicação com o servidor.' });
        } else {
            alert('Erro de comunicação com o servidor.');
        }
    });
}

// Inicialização
$(document).ready(function() {
    // Aplicar máscara nos campos de valor existentes
    $('.valor-input').each(function() {
        VendasModal.aplicarMascaraValorInline($(this));
    });
    
    // Event delegation para edição inline - funciona com elementos criados dinamicamente
    // Usando delegate mais específico no tbody para melhor performance
    $('#tabelaVendas').on('click', 'tbody tr .editable-cell', function(e) {
        console.log('🎯 CLICK EVENT - Editable Cell detectado (delegate específico)!');
        console.log('🔍 Target:', e.target);
        console.log('🔍 CurrentTarget:', e.currentTarget);
        
        // Verificar se não está clicando em um input já aberto
        if ($(e.target).is('input') || $(e.target).closest('input').length > 0) {
            console.log('🚫 Clique em input detectado, ignorando...');
            return;
        }
        
        // Verificar se a linha já está em modo de edição
        const linha = $(this).closest('tr');
        if (linha.hasClass('editing-row')) {
            console.log('🚫 Linha já em modo de edição, ignorando...');
            return;
        }
        
        // Obter o ID da venda
        const idVenda = linha.data('id') || linha.attr('data-id');
        console.log('🔍 ID da venda detectado:', idVenda);
        console.log('🔍 Tipo do ID:', typeof idVenda);
        console.log('🔍 String do ID:', String(idVenda));
        console.log('🔍 Starts with novo_:', String(idVenda).startsWith('novo_'));
        
        // Verificar se é uma linha válida (não é nova linha)
        if (idVenda && !String(idVenda).startsWith('novo_') && String(idVenda) !== 'Novo') {
            console.log('✅ Iniciando edição inline para ID:', idVenda);
            VendasModal.editarInline(idVenda);
        } else {
            console.log('🚫 ID inválido ou linha temporária, não editando. ID:', idVenda);
        }
    });
    
    // Event delegation para botões de ação também
    $('#tabelaVendas').on('click', 'tbody tr .btn-edit', function(e) {
        console.log('🔧 CLICK EVENT - Botão editar detectado (delegate específico)!');
        e.preventDefault();
        e.stopPropagation();
        const linha = $(this).closest('tr');
        const idVenda = linha.data('id') || linha.attr('data-id');
        console.log('🔍 ID da venda (botão):', idVenda);
        console.log('🔍 Tipo do ID (botão):', typeof idVenda);
        
        if (idVenda && !String(idVenda).startsWith('novo_') && String(idVenda) !== 'Novo') {
            console.log('✅ Iniciando edição via botão para ID:', idVenda);
            VendasModal.editarInline(idVenda);
        } else {
            console.log('🚫 ID inválido via botão, não editando. ID:', idVenda);
        }
    });
    
    // Inicializar DataTable com filtros por coluna
    if ($.fn.DataTable && $('#tabelaVendas').length > 0) {
        VendasModal.criarDataTable();
    }
});
</script>